#pragma once

#include "session_factory.h"
#include "session_storage.h"
#include "config.h"
#include "helpers.h"
#include "stats.h"

#include <util/generic/deque.h>
#include <util/system/condvar.h>
#include <util/system/mutex.h>
#include <util/system/thread.h>
#include <util/system/event.h>

namespace NCaptchaServer {
    struct TCachedSessionInfo {
        TString Token;
        TCaptchaSessionInfo SessionInfo;

        TCachedSessionInfo() {
        }

        TCachedSessionInfo(const TString& token, TCaptchaSessionInfo&& sessionInfo)
            : Token(token)
            , SessionInfo(sessionInfo)
        {
        }
    };

    class TCaptchaPrecachingSessionStorage: public ICaptchaSessionStorage {
    public:
        TCaptchaPrecachingSessionStorage(const TCaptchaConfig& config, TCaptchaStats& stats, ICaptchaSessionStorage* slave);

        char TokenTag() override;
        NThreading::TFuture<TString> CreateSession(const TCaptchaSessionRequest& request, TCaptchaSessionInfo& info) override;
        NThreading::TFuture<bool> LoadSessionInfo(TStringBuf token, TCaptchaSessionInfo& info) override;
        NThreading::TFuture<bool> StoreSessionInfo(TStringBuf token, const TCaptchaSessionInfo& info) override;
        NThreading::TFuture<void> DropSession(TStringBuf token) override;

        ~TCaptchaPrecachingSessionStorage();

    private:
        using TCacheKey = std::tuple<TString, TString, ui32>;

        class TSessionCache {
        public:
            TSessionCache(TCaptchaStats& stats, TMutex& cacheItemsMutex, TCondVar& cacheItemDropCondVar, const TCacheKey& key, ui32 maxSessions);
            bool IsFull() const;
            ui32 Size() const;
            ui32 UnfilledCount() const;
            TInstant OldestTimestamp() const;
            void AddSession(TCachedSessionInfo&& session);
            bool GetSession(TInstant notBefore, TCachedSessionInfo& session);
            void DropExpiredSessions(TInstant notBefore);

        private:
            void DoDropExpiredSessions(TInstant notBefore);

        private:
            TCaptchaStats& Stats;
            TMutex& CacheItemsMutex;
            TCondVar& CacheItemDropCondVar;

            TString CaptchaType;
            TString VoiceType;
            ui32 Checks;
            ui32 MaxSessions;
            TDeque<TCachedSessionInfo> Cache;
        };

    private:
        static void* RefillLoop(void* ptr);

        void AddCache(const TCacheKey& key, ui32 maxSessions);
        void DropExpiredSessions();
        void PrepareNewSession(const TCacheKey& key);
        ui32 LoadNewSessions(ui32 count);
        ui32 TotalCacheSize() const;
        TInstant NextExpiration() const;

    private:
        TResourceAvailabilityMonitor AvailabilityMonitor;
        TResourceProviderGuard ProviderGuard;

        TCaptchaConfig Config;
        TCaptchaStats& Stats;
        THolder<ICaptchaSessionStorage> Slave;

        THashMap<TString, TString> CaptchaTypeAliases;
        THashMap<TString, TString> VoiceCaptchaTypeAliases;

        TDuration SessionReservedTime;
        ui32 MaxAwaitingResponse;
        ui32 MaxCacheSize;

        THashMap<TCacheKey, TSessionCache> Caches;
        TVector<TCacheKey> CacheKeys;

        TMutex CacheItemsMutex;
        TCondVar CacheItemDropCondVar;

        TMutex AwaitingResponseMutex;
        TCondVar ReceivedResponseCondVar;
        ui32 AwaitingResponse = 0;

        size_t CachesRoundRobin = 0;

        TAtomic Exiting;
        TThread RefillerThread;
    };
}
