#pragma once

#include "items_storage_router.h"

#include <library/cpp/json/json_value.h>
#include <library/cpp/threading/future/future.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>
#include <util/stream/str.h>

#include <time.h>

namespace NCaptchaServer {
    struct TCaptchaSessionRequest {
        TStringBuf Type;
        TStringBuf VoiceType;
        int Checks;
        TInstant Timestamp;

        TCaptchaSessionRequest(TStringBuf type, TStringBuf voiceType, int checks, TInstant timestamp)
            : Type(type)
            , VoiceType(voiceType)
            , Checks(checks)
            , Timestamp(timestamp)
        {
        }
    };

    struct TCaptchaSessionInfo {
        TString Type;
        NJson::TJsonValue Metadata;
        TInstant Timestamp;
        int Checks = 1;
        bool Fallback = false;
        ECaptchaItemsStorageId ItemsStorage;

        TString DebugStr() const {
            TString result;
            TStringOutput so(result);
            so << Type.Quote() << ";" << Metadata << ";" << Timestamp.ToString() << ";" << Checks << ";" << Fallback << ";" << ItemsStorage;
            return result;
        }
    };

    class ICaptchaSessionStorage {
    public:
        virtual char TokenTag() = 0; // second character in a token
        virtual NThreading::TFuture<TString> CreateSession(const TCaptchaSessionRequest& request, TCaptchaSessionInfo& info) = 0;
        virtual NThreading::TFuture<bool> LoadSessionInfo(TStringBuf token, TCaptchaSessionInfo& info) = 0;
        virtual NThreading::TFuture<bool> StoreSessionInfo(TStringBuf token, const TCaptchaSessionInfo& info) = 0;
        virtual NThreading::TFuture<void> DropSession(TStringBuf token) = 0;
        virtual ~ICaptchaSessionStorage() {
        }

        NThreading::TFuture<bool> LoadAndValidateSessionInfo(TStringBuf token, TCaptchaSessionInfo& info, TDuration timeout) {
            auto cont = [timeout, &info](const NThreading::TFuture<bool>& fresult) {
                bool result = fresult.GetValue();
                if (!result) {
                    return false;
                }
                if (!info.Fallback && Now() - info.Timestamp > timeout) {
                    return false;
                }
                return true;
            };
            return LoadSessionInfo(token, info).Apply(cont);
        }
    };

    constexpr size_t TokenLength = 32;
}
