#pragma once

#include "session_storage.h"
#include "fallback_session_storage.h"

#include <library/cpp/logger/global/global.h>

#include <util/generic/xrange.h>
#include <util/generic/hash.h>

namespace NCaptchaServer {
    class TCaptchaSessionStorageRouter {
    public:
        void RegisterMainStorage(ICaptchaSessionStorage* storage) {
            MainStorage = storage;
            MainStorageTag = storage->TokenTag();
            Storages[MainStorageTag] = THolder<ICaptchaSessionStorage>(MainStorage);
        }

        void RegisterSecondaryStorage(ICaptchaSessionStorage* storage) {
            Storages[storage->TokenTag()] = THolder<ICaptchaSessionStorage>(storage);
        }

        void RegisterFallbackStorage(TCaptchaFallbackSessionStorage* storage) {
            FallbackStorage = storage;
            FallbackStorageTag = storage->TokenTag();
            Storages[FallbackStorageTag] = THolder<ICaptchaSessionStorage>(FallbackStorage);
        }

        ICaptchaSessionStorage* GetMainStorage() {
            if (FallbackActive) {
                Y_ASSERT(FallbackStorage);
                return FallbackStorage;
            }

            if (MainStorage) {
                return MainStorage;
            }
            ythrow yexception() << "Main storage is not registered";
        }

        TCaptchaFallbackSessionStorage* GetFallbackStorage() {
            return FallbackStorage;
        }

        ICaptchaSessionStorage* GetStorageByToken(TStringBuf token) {
            if (token.size() < 2) {
                return nullptr;
            }
            auto iter = Storages.find(token[1]);
            if (iter == Storages.end()) {
                return nullptr;
            }
            return iter->second.Get();
        }

        bool IsFallbackStorage(ICaptchaSessionStorage* storage) const {
            return FallbackStorage == storage;
        }

        bool IsFallbackActive() const {
            return FallbackActive;
        }

        void SetFallbackState(bool active) {
            if (active && !FallbackActive) {
                Y_ENSURE(FallbackStorage);
                NOTICE_LOG << "Activating fallback" << Endl;
            } else if (!active && FallbackActive) {
                NOTICE_LOG << "Deactivating fallback" << Endl;
                Y_ENSURE(MainStorage);
            }
            FallbackActive = active;
        }

    private:
        bool FallbackActive = false;
        char MainStorageTag = 0;
        char FallbackStorageTag = 0;
        ICaptchaSessionStorage* MainStorage = nullptr;
        TCaptchaFallbackSessionStorage* FallbackStorage = nullptr;
        THashMap<char, THolder<ICaptchaSessionStorage>> Storages;
    };
}
