#include "stats.h"
#include "helpers.h"

#include <util/generic/xrange.h>
#include <util/string/printf.h>
#include <util/string/subst.h>

namespace NCaptchaServer {
    static TString NormalizeHandlerName(TStringBuf handler) {
        Y_ASSERT(handler[0] == '/');
        handler.Skip(1);
        TString result = TString(handler);
        SubstGlobal(result, "/", "_");
        return result;
    }

    static TString NormalizeTypeName(TStringBuf name) {
        if (name == "") {
            return "default";
        }
        return TString(name);
    }

    static const NUnistat::TIntervals TimingsIntervals = {0, 1, 5, 10, 20, 35, 50, 80, 100, 160, 200, 300, 600, 1000};
    static const NUnistat::TIntervals AnswerTimesIntervals = {0, 1000, 5000, 10000, 30000, 60000, 120000, 300000, 600000, 1200000};

    TCaptchaStats::TCaptchaStats(TCaptchaConfig& config) {
        THashMap<TString, TString> voiceCaptchaTypeAliases;

        FillTypeAliases(config.GetCaptchaTypes(), CaptchaTypeAliases);
        FillTypeAliases(config.GetVoiceCaptchaTypes(), voiceCaptchaTypeAliases);

        for (const auto& bucket : config.GetSessionCache().GetBuckets()) {
            const auto& key = bucket.GetKey();
            const TString& ctype = GetWithDefault(CaptchaTypeAliases, key.GetType());
            const TString& vtype = GetWithDefault(voiceCaptchaTypeAliases, key.GetVoiceType());
            SessionCacheKeys.push_back(std::make_tuple(ctype, vtype, key.GetChecks()));
        }

        InitSignals();
    }

    void TCaptchaStats::InitHandlerSegmentedSignals(const TDeque<TString>& handlers) {
        for (const auto& handler : handlers) {
            auto drillHandlerSegmentedFloatHole = [&](EHandlerSegmentedSignals signal) {
                TString signame = Sprintf(ToString(signal).c_str(), NormalizeHandlerName(handler).c_str());
                Unistat.DrillFloatHole(signame, "summ", NUnistat::TPriority(0));
            };

            auto drillHandlerSegmentedHistogramHole = [&](EHandlerSegmentedSignals signal, const NUnistat::TIntervals& intervals) {
                TString signame = Sprintf(ToString(signal).c_str(), NormalizeHandlerName(handler).c_str());
                Unistat.DrillHistogramHole(signame, "hgram", NUnistat::TPriority(0), intervals);
            };

            drillHandlerSegmentedFloatHole(EHandlerSegmentedSignals::Requests);
            drillHandlerSegmentedHistogramHole(EHandlerSegmentedSignals::ResponseTimingsMs, TimingsIntervals);
        }
    }

    void TCaptchaStats::InitSignals() {
        Unistat.DrillFloatHole(ToString(ESignals::TotalRequests), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::TotalExceptions), "summ", NUnistat::TPriority(0));

        Unistat.DrillHistogramHole(ToString(ESignals::ResponseTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);

        Unistat.DrillFloatHole(ToString(ESignals::ChecksOk), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::ChecksFail), "summ", NUnistat::TPriority(0));

        Unistat.DrillHistogramHole(ToString(ESignals::AnswerTimeMsOk), "hgram", NUnistat::TPriority(0), AnswerTimesIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::AnswerTimeMsFail), "hgram", NUnistat::TPriority(0), AnswerTimesIntervals);

        Unistat.DrillFloatHole(ToString(ESignals::TotalRequestsFormatJson), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::TotalRequestsFormatXml), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::ItemCacheHits), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::ItemCacheMisses), "summ", NUnistat::TPriority(0));

        Unistat.DrillFloatHole(ToString(ESignals::ExpiredSessionsRemoved), "summ", NUnistat::TPriority(0));

        Unistat.DrillFloatHole(ToString(ESignals::StorageFallbackRequestsGenerate), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::StorageFallbackRequestsImage), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::StorageFallbackRequestsVoice), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::StorageFallbackRequestsCheck), "summ", NUnistat::TPriority(0));

        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryCreateSessionTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryLoadSessionInfoTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryStoreSessionInfoTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryDropSessionTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryCleanSessionsTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryLoadItemTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryLoadAllItemTypeVersionsTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::KikimrQueryLoadSingleItemTypeVersionTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);
        Unistat.DrillHistogramHole(ToString(ESignals::WriteResponseTimingsMs), "hgram", NUnistat::TPriority(0), TimingsIntervals);

        Unistat.DrillFloatHole(ToString(ESignals::SessionPrecachingExpired), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::SessionPrecachingHits), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::SessionPrecachingDefinedMisses), "summ", NUnistat::TPriority(0));
        Unistat.DrillFloatHole(ToString(ESignals::SessionPrecachingMisses), "summ", NUnistat::TPriority(0));

        for (auto alias : CaptchaTypeAliases) {
            TString type = NormalizeTypeName(alias.first);

            auto drillTypeSegmentedFloatHole = [&](ETypeSegmentedSignals signal) {
                TString signame = Sprintf(ToString(signal).c_str(), type.c_str());
                Unistat.DrillFloatHole(signame, "summ", NUnistat::TPriority(0));
            };

            auto drillTypeSegmentedHistogramHole = [&](ETypeSegmentedSignals signal, const NUnistat::TIntervals& intervals) {
                TString signame = Sprintf(ToString(signal).c_str(), type.c_str());
                Unistat.DrillHistogramHole(signame, "hgram", NUnistat::TPriority(0), intervals);
            };

            drillTypeSegmentedFloatHole(ETypeSegmentedSignals::ChecksOk);
            drillTypeSegmentedFloatHole(ETypeSegmentedSignals::ChecksFail);
            drillTypeSegmentedFloatHole(ETypeSegmentedSignals::RequestsGenerate);

            drillTypeSegmentedHistogramHole(ETypeSegmentedSignals::AnswerTimeMsOk, AnswerTimesIntervals);
            drillTypeSegmentedHistogramHole(ETypeSegmentedSignals::AnswerTimeMsFail, AnswerTimesIntervals);
        }

        for (const auto& key : SessionCacheKeys) {
            auto drillSessionCacheKeySegmentedFloatHole = [&](ESessionCacheKeySegmentedSignals signal) {
                TString signame = Sprintf(ToString(signal).c_str(), std::get<0>(key).c_str(), std::get<1>(key).c_str(), unsigned(std::get<2>(key)));
                Unistat.DrillFloatHole(signame, "summ", NUnistat::TPriority(0));
            };

            drillSessionCacheKeySegmentedFloatHole(ESessionCacheKeySegmentedSignals::SessionPrecachingExpired);
        }
    }

    void TCaptchaStats::RegisterSignalCallback(ESignals signal, std::function<double()> callback, const TString& suffix) {
        Callbacks.emplace_back(ToString(signal), callback, suffix);
        Unistat.DrillFloatHole(ToString(signal), suffix, NUnistat::TPriority(0), NUnistat::TStartValue(0), EAggregationType::LastValue);
    }

    void TCaptchaStats::RegisterSignalCallback(ESessionCacheKeySegmentedSignals signal, std::function<double(const TString&, const TString&, ui32)> callback, const TString& suffix) {
        for (const auto& key : SessionCacheKeys) {
            TString ctype = std::get<0>(key);
            TString vtype = std::get<1>(key);
            ui32 checks = std::get<2>(key);

            TString signame = Sprintf(ToString(signal).c_str(), ctype.c_str(), vtype.c_str(), unsigned(checks));

            auto wrapper = [callback, ctype, vtype, checks]() {
                return callback(ctype, vtype, checks);
            };

            Callbacks.emplace_back(signame, wrapper, suffix);
            Unistat.DrillFloatHole(signame, suffix, NUnistat::TPriority(0), NUnistat::TStartValue(0), EAggregationType::LastValue);
        }
    }

    void TCaptchaStats::PushSignal(ESignals signal, double value) {
        Unistat.PushSignalUnsafe(signal, value);
    }

    void TCaptchaStats::PushSignal(ETypeSegmentedSignals signal, TStringBuf type, double value) {
        if (!CaptchaTypeAliases.contains(type)) {
            type = TStringBuf();
        }
        TString signame = Sprintf(ToString(signal).c_str(), NormalizeTypeName(type).c_str());
        Unistat.PushSignalUnsafe(signame, value);
    }

    void TCaptchaStats::PushSignal(EHandlerSegmentedSignals signal, TStringBuf handler, double value) {
        TString signame = Sprintf(ToString(signal).c_str(), NormalizeHandlerName(handler).c_str());
        Unistat.PushSignalUnsafe(signame, value);
    }

    void TCaptchaStats::PushSignal(ESessionCacheKeySegmentedSignals signal, TStringBuf ctype, TStringBuf vtype, ui32 checks, double value) {
        TString signame = Sprintf(ToString(signal).c_str(), ctype.data(), vtype.data(), unsigned(checks));
        Unistat.PushSignalUnsafe(signame, value);
    }

    TString TCaptchaStats::CreateJsonDump() {
        UpdateCallbackSignals();
        return Unistat.CreateJsonDump(0, true);
    }

    void TCaptchaStats::UpdateCallbackSignals() {
        for (const auto& callbackInfo : Callbacks) {
            Unistat.PushSignalUnsafe(callbackInfo.Signal, callbackInfo.Callback());
        }
    }

}
