#pragma once

#include "config.h"

#include <library/cpp/unistat/unistat.h>
#include <util/generic/deque.h>

namespace NCaptchaServer {
    //NOTE: when adding new signals, don't forget to init hole in TCaptchaStats::InitSignals
    enum class ESignals {
        TotalRequests /* "total_requests" */,
        TotalExceptions /* "total_exceptions" */,
        ResponseTimingsMs /* "response_timings_ms" */,
        ChecksOk /* "checks_ok" */,
        ChecksFail /* "checks_fail" */,
        AnswerTimeMsOk /* "answer_time_ms_ok" */,
        AnswerTimeMsFail /* "answer_time_ms_fail" */,
        TotalRequestsFormatJson /* "total_requests_format_json" */,
        TotalRequestsFormatXml /* "total_requests_format_xml" */,
        ItemCacheHits /* "item_cache_hits" */,
        ItemCacheMisses /* "item_cache_misses" */,
        KikimrQueryCreateSessionTimingsMs /* "kikimr_query_create_session_timings_ms" */,
        KikimrQueryLoadSessionInfoTimingsMs /* "kikimr_query_load_session_info_timings_ms" */,
        KikimrQueryStoreSessionInfoTimingsMs /* "kikimr_query_store_session_info_timings_ms" */,
        KikimrQueryDropSessionTimingsMs /* "kikimr_query_drop_session_timings_ms" */,
        KikimrQueryCleanSessionsTimingsMs /* "kikimr_query_clean_sessions_timings_ms" */,
        KikimrQueryLoadItemTimingsMs /* "kikimr_query_load_item_timings_ms" */,
        KikimrQueryLoadAllItemTypeVersionsTimingsMs /* "kikimr_query_load_all_item_type_versions_timings_ms" */,
        KikimrQueryLoadSingleItemTypeVersionTimingsMs /* "kikimr_query_load_single_item_type_version_timings_ms" */,
        KikimrItemClientYdbSessions /* "kikimr_item_client_ydb_sessions" */,
        KikimrSessionClientYdbSessions /* "kikimr_session_client_ydb_sessions" */,
        RequestQueueSize /* "request_queue_size" */,
        ConnectionCount /* "connection_count" */,
        WriteResponseTimingsMs /* "write_response_timings_ms" */,
        ExpiredSessionsRemoved /* "expired_sessions_removed" */,
        StorageFallbackRequestsGenerate /* "storage_fallback_requests_generate" */,
        StorageFallbackRequestsImage /* "storage_fallback_requests_image" */,
        StorageFallbackRequestsVoice /* "storage_fallback_requests_voice" */,
        StorageFallbackRequestsCheck /* "storage_fallback_requests_check" */,
        TimeSinceLastIndexUpdateMs /* "time_since_last_index_update_ms" */,
        SessionPrecachingExpired /* "session_precaching_expired" */,
        SessionPrecachingCacheSize /* "session_precaching_cache_size" */,
        SessionPrecachingHits /* "session_precaching_hits" */,
        SessionPrecachingDefinedMisses /* "session_precaching_defined_misses" */,
        SessionPrecachingMisses /* "session_precaching_misses" */,
        SessionPrecachingAwaitingResponse /* "session_precaching_awaiting_response" */,
        SessionPrecachingTimeSinceOldestSessionMs /* "session_precaching_time_since_oldest_session_ms" */,
    };

    enum class ETypeSegmentedSignals {
        ChecksOk /* "checks_ok_type_%s" */,
        ChecksFail /* "checks_fail_type_%s" */,
        RequestsGenerate /* "requests_generate_type_%s" */,
        AnswerTimeMsOk /* "answer_time_ms_ok_type_%s" */,
        AnswerTimeMsFail /* "answer_time_ms_fail_type_%s" */,
    };

    enum class EHandlerSegmentedSignals {
        Requests /* "requests_handler_%s" */,
        ResponseTimingsMs /* "response_timings_ms_%s" */,
    };

    enum class ESessionCacheKeySegmentedSignals {
        SessionPrecachingExpired /* "session_precaching_expired_%s_%s_%u" */,
        SessionPrecachingTimeSinceOldestSessionMs /* "session_precaching_time_since_oldest_session_ms_%s_%s_%u" */,
    };

    class TCaptchaStats {
    public:
        TCaptchaStats(TCaptchaConfig& config);
        void InitHandlerSegmentedSignals(const TDeque<TString>& handlers);
        void RegisterSignalCallback(ESignals signal, std::function<double()> callback, const TString& suffix = "max");
        void RegisterSignalCallback(ESessionCacheKeySegmentedSignals signal, std::function<double(const TString&, const TString&, ui32)> callback, const TString& suffix = "max");

        void PushSignal(ESignals signal, double value = 1.0);
        void PushSignal(ETypeSegmentedSignals signal, TStringBuf type, double value = 1.0);
        void PushSignal(EHandlerSegmentedSignals signal, TStringBuf handler, double value = 1.0);
        void PushSignal(ESessionCacheKeySegmentedSignals signal, TStringBuf ctype, TStringBuf vtype, ui32 checks, double value = 1.0);
        TString CreateJsonDump();

    private:
        void InitSignals();
        void UpdateCallbackSignals();

    private:
        friend class TCaptchaStatsInitializer;
        struct TSignalCallback {
            TString Signal;
            std::function<double()> Callback;
            TString Suffix;

            TSignalCallback(const TString& signal, std::function<double()> callback, const TString& suffix)
                : Signal(signal)
                , Callback(callback)
                , Suffix(suffix)
            {
            }
        };

    private:
        TUnistat Unistat;
        TDeque<TSignalCallback> Callbacks;
        THashMap<TString, TString> CaptchaTypeAliases;
        TDeque<std::tuple<TString, TString, ui32>> SessionCacheKeys;
    };
}
