#include <captcha/server/lib/http_server.h>
#include <captcha/server/lib/session_factory.h>
#include <captcha/server/lib/items_storage_router.h>
#include <captcha/server/lib/session_storage_router.h>
#include <captcha/server/lib/caching_items_storage.h>
#include <captcha/server/lib/precaching_session_storage.h>
#include <captcha/server/lib/fallback_items_storage.h>
#include <captcha/server/lib/fallback_session_storage.h>
#include <captcha/server/lib/fallback_state_aggregator.h>
#include <captcha/server/lib/mode_file_watcher.h>

#include <captcha/server/ydb_storage/ydb_items_storage.h>
#include <captcha/server/ydb_storage/ydb_session_storage.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/logger/thread.h>
#include <library/cpp/sighandler/async_signals_handler.h>
#include <util/folder/path.h>

struct TOptions {
    TFsPath ConfigPath;
};

TOptions ParseArguments(int argc, char** argv) {
    TOptions result;
    auto opts = NLastGetopt::TOpts::Default();

    opts.AddLongOption('c', "config")
        .StoreResult(&result.ConfigPath)
        .Required()
        .Help("Path to server config.");

    opts.SetFreeArgsNum(0);
    NLastGetopt::TOptsParseResult args{&opts, argc, argv};
    return result;
}

void InitStorageRouters(const NCaptchaServer::TCaptchaConfig& config, NCaptchaServer::TCaptchaStats& stats, NCaptchaServer::TCaptchaSessionFactory& sessionFactory, NCaptchaServer::TCaptchaItemsStorageRouter& itemsRouter, NCaptchaServer::TCaptchaSessionStorageRouter& sessRouter, NCaptchaServer::TCaptchaFallbackStateAggregator& fallbackStateAggregator) {
    if (config.GetFallback().GetActive() || config.GetFallback().GetLoaded()) {
        NCaptchaServer::TCaptchaFallbackItemsStorage* fallbackItemsStorage = new NCaptchaServer::TCaptchaFallbackItemsStorage(config, stats);
        NCaptchaServer::TCaptchaFallbackSessionStorage* sessionStorage = new NCaptchaServer::TCaptchaFallbackSessionStorage(config, stats, fallbackItemsStorage);

        itemsRouter.RegisterStorage(NCaptchaServer::ECaptchaItemsStorageId::Fallback, fallbackItemsStorage);
        sessRouter.RegisterFallbackStorage(sessionStorage);
    }

    if (!config.GetFallback().GetActive() || !config.GetFallback().GetOnlyFallback()) {
        NCaptchaServer::TCaptchaYdbItemsStorage* backendItemsStorage = new NCaptchaServer::TCaptchaYdbItemsStorage(config, stats);
        NCaptchaServer::TCaptchaCachingItemsStorage* cachedItemsStorage = new NCaptchaServer::TCaptchaCachingItemsStorage(config, stats, backendItemsStorage);

        THolder<NCaptchaServer::ICaptchaSessionStorage> sessionStorage = MakeHolder<NCaptchaServer::TCaptchaYdbSessionStorage>(config, stats, sessionFactory);
        if (config.GetSessionCache().GetEnabled()) {
            sessionStorage = MakeHolder<NCaptchaServer::TCaptchaPrecachingSessionStorage>(config, stats, sessionStorage.Release());
        }

        itemsRouter.RegisterStorage(NCaptchaServer::ECaptchaItemsStorageId::Kikimr, cachedItemsStorage);
        sessRouter.RegisterMainStorage(sessionStorage.Release());
    }

    fallbackStateAggregator.SetFallbackState(NCaptchaServer::ECaptchaFallbackActivator::ConfigOrFallbackHandler, config.GetFallback().GetActive());
}

int main(int argc, char** argv) {
    auto options = ParseArguments(argc, argv);

    NCaptchaServer::TCaptchaConfig config(options.ConfigPath);

    DoInitGlobalLog(config.GetServiceLogPath(), FromString<ELogPriority>(config.GetServiceLogLevel()), false, false);
    auto& log = TLoggerOperator<TGlobalLog>::Log();
    log.ResetBackend(THolder(new TOwningThreadedLogBackend(log.ReleaseBackend().Release())));

    NCaptchaServer::TCaptchaStats stats(config);

    NCaptchaServer::TCaptchaItemsStorageRouter itemsStorageRouter;
    NCaptchaServer::TCaptchaSessionFactory sessionFactory(config, stats, itemsStorageRouter);

    NCaptchaServer::TCaptchaSessionStorageRouter sessionStorageRouter;
    NCaptchaServer::TCaptchaFallbackStateAggregator fallbackStateAggregator(sessionStorageRouter);

    InitStorageRouters(config, stats, sessionFactory, itemsStorageRouter, sessionStorageRouter, fallbackStateAggregator);

    THolder<NCaptchaServer::TCaptchaModeFileWatcher> modeFileWatcher;
    if (config.GetModeFileWatcher().GetPath()) {
        modeFileWatcher = MakeHolder<NCaptchaServer::TCaptchaModeFileWatcher>(config, fallbackStateAggregator);
    }

    NCaptchaServer::TCaptchaHttpServer server(config, stats, sessionFactory, itemsStorageRouter, sessionStorageRouter, fallbackStateAggregator);

    SetAsyncSignalFunction(SIGTERM, [&server](int) { server.Shutdown(); });

    server.Start();
    NOTICE_LOG << "Captcha server started" << Endl;
    server.Wait();
    NOTICE_LOG << "Captcha server stopped" << Endl;
}
