#pragma once

#include <captcha/server/lib/config.h>
#include <captcha/server/lib/stats.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <util/string/strip.h>
#include <util/system/thread.h>
#include <util/system/event.h>

namespace NCaptchaServer {
    using TYdbDatabaseOptions = NCaptchaServerConfigProto::TKikimrDatabase;

    class TCaptchaYdbClient {
    public:
        TCaptchaYdbClient(const TYdbDatabaseOptions& options, TCaptchaStats& stats);

        template <typename TParamsBuilderFunc>
        NYdb::NTable::TAsyncDataQueryResult RunQuery(ESignals signal, const TString& query, TParamsBuilderFunc paramsBuilderFunc, TDuration timeout, NYdb::NTable::TTxSettings txSettings = NYdb::NTable::TTxSettings::SerializableRW());

        NYdb::NTable::TTableClient& GetClient();
        const TString& GetDatabaseName();

    private:
        TCaptchaStats& Stats;

        NYdb::TDriver Driver;
        NYdb::NTable::TTableClient Client;
        const TString DatabaseName;
    };

    inline TString QueryDescription(const TString& query) {
        // Assuming that first line of the query contains comment with description.

        TString stripped = StripString(query);
        return TString{TStringBuf(stripped).Before('\n')}.Quote();
    }

    template <typename TParamsBuilderFunc>
    NYdb::NTable::TAsyncDataQueryResult TCaptchaYdbClient::RunQuery(ESignals signal, const TString& query, TParamsBuilderFunc paramsBuilderFunc, TDuration timeout, NYdb::NTable::TTxSettings txSettings) {
        TInstant start = Now();

        auto fsresult = Client.GetSession(NYdb::NTable::TCreateSessionSettings().ClientTimeout(timeout));
        auto result = fsresult.Apply([query, timeout, paramsBuilderFunc, start, txSettings](const NThreading::TFuture<NYdb::NTable::TCreateSessionResult>& fsresult) {
            const auto& sresult = fsresult.GetValue();
            if (!sresult.IsSuccess()) {
                ythrow yexception() << "YDB Error while creating session for query " << QueryDescription(query) << ": " << sresult.GetStatus() << " " << sresult.GetIssues().ToString();
            }
            auto session = sresult.GetSession();

            const auto elapsedAfterGetSession = Now() - start;
            if (elapsedAfterGetSession >= timeout) {
                ythrow yexception() << "YDB Error after creating session for query " << QueryDescription(query) << ": Timed out";
            }

            auto fpresult = session.PrepareDataQuery(query, NYdb::NTable::TPrepareDataQuerySettings().ClientTimeout(timeout - elapsedAfterGetSession));
            return fpresult.Apply([query, timeout, paramsBuilderFunc, start, txSettings](const NYdb::NTable::TAsyncPrepareQueryResult& fpresult) {
                const auto& presult = fpresult.GetValue();
                if (!presult.IsSuccess()) {
                    ythrow yexception() << "YDB Error while preparing query " << QueryDescription(query) << ": " << presult.GetStatus() << " " << presult.GetIssues().ToString();
                }

                auto preparedQuery = presult.GetQuery();

                const auto elapsedAfterPrepare = Now() - start;
                if (elapsedAfterPrepare >= timeout) {
                    ythrow yexception() << "YDB Error after preparing query " << QueryDescription(query) << ": Timed out";
                }

                return preparedQuery.Execute(NYdb::NTable::TTxControl::BeginTx(txSettings).CommitTx(),
                                             paramsBuilderFunc(preparedQuery.GetParamsBuilder()),
                                             NYdb::NTable::TExecDataQuerySettings().ClientTimeout(timeout - elapsedAfterPrepare));
            });
        });

        return result.Apply([this, signal, query, start](const NYdb::NTable::TAsyncDataQueryResult& fqresult) {
            Stats.PushSignal(signal, (Now() - start).MilliSeconds());

            const auto& qresult = fqresult.GetValue();
            if (!qresult.IsSuccess()) {
                ythrow yexception() << "YDB Error while executing query " << QueryDescription(query) << ": " << qresult.GetStatus() << " " << qresult.GetIssues().ToString();
            }
            return fqresult;
        });
    }
}
