#include "ydb_items_storage.h"

#include <library/cpp/json/json_reader.h>
#include <util/generic/xrange.h>
#include <util/string/printf.h>

namespace NCaptchaServer {
    TCaptchaYdbItemsStorage::TCaptchaYdbItemsStorage(const TCaptchaConfig& config, TCaptchaStats& stats)
        : Client(config.GetItems().GetDatabase(), stats)
    {
        LoadDataTimeout = TDuration::MilliSeconds(config.GetTimeouts().GetItemsClientTimeoutMs());
        LoadVersionsTimeout = TDuration::MilliSeconds(config.GetTimeouts().GetLoadItemVersionsQueryMs());
        ReadItemIndexTimeout = TDuration::MilliSeconds(config.GetTimeouts().GetReadItemIndexMs());

        TString itemsTable = config.GetItems().GetItemsTable();
        TString versionsTable = config.GetItems().GetVersionsTable();

        const auto& dbName = config.GetItems().GetDatabase().GetName();
        ItemsTableFullPath = dbName + "/" + itemsTable;

        const char* loadItemDataQueryTemplate = R"___(
            -- Load item data
            declare $type as String;
            declare $version as String;
            declare $id as String;
            select data, metadata from [%s] where type = $type and version = $version and id = $id
            )___";
        LoadItemDataQuery = Sprintf(loadItemDataQueryTemplate, itemsTable.c_str());

        const char* currentVersionsQueryTemplate = R"___(
            -- Load current versions for all types
            select times.type as type, version
            from [%s]
            inner join (
                select type, max(timestamp) as timestamp from [%s] group by type
            ) as times
            using (type, timestamp)
            )___";
        CurrentVersionsQuery = Sprintf(currentVersionsQueryTemplate, versionsTable.c_str(), versionsTable.c_str());

        const char* currentVersionSingleQueryTemplate = R"___(
            -- Load current version for a single type
            declare $type as String;
            select version, timestamp from [%s] where type = $type order by timestamp desc limit 1
            )___";
        CurrentVersionSingleQuery = Sprintf(currentVersionSingleQueryTemplate, versionsTable.c_str());

        stats.RegisterSignalCallback(ESignals::KikimrItemClientYdbSessions, [this]() { return Client.GetClient().GetActiveSessionCount(); });
    }

    NThreading::TFuture<bool> TCaptchaYdbItemsStorage::LoadItemData(const TCaptchaItemKey& key, TString& data) {
        auto paramBuilder = [key, data](NYdb::TParamsBuilder&& builder) {
            builder.AddParam("$type").String(key.Type).Build();
            builder.AddParam("$version").String(key.Version).Build();
            builder.AddParam("$id").String(key.Id).Build();
            return builder.Build();
        };

        return Client.RunQuery(ESignals::KikimrQueryLoadItemTimingsMs, LoadItemDataQuery, paramBuilder, LoadDataTimeout).Apply([&data](const NYdb::NTable::TAsyncDataQueryResult& fqresult) {
            auto qresult = fqresult.GetValue();
            Y_ENSURE(qresult.GetResultSets().size() == 1);

            auto parser = qresult.GetResultSetParser(0);

            if (parser.TryNextRow()) {
                data = parser.ColumnParser("data").GetOptionalString().GetRef();

                Y_ENSURE(!parser.TryNextRow());
                return true;
            }

            return false;
        });
    }

    void TCaptchaYdbItemsStorage::LoadIndex(TCaptchaItemsIndex& index) {
        auto noparams = [](NYdb::TParamsBuilder&& builder) {
            return builder.Build();
        };

        auto versions = Client.RunQuery(ESignals::KikimrQueryLoadAllItemTypeVersionsTimingsMs, CurrentVersionsQuery, noparams, LoadVersionsTimeout).GetValue(LoadVersionsTimeout);
        Y_ENSURE(versions.GetResultSets().size() == 1);
        auto parser = versions.GetResultSetParser(0);

        index.clear();
        while (parser.TryNextRow()) {
            TString type = parser.ColumnParser("type").GetOptionalString().GetRef();
            TString version = parser.ColumnParser("version").GetOptionalString().GetRef();
            ExtendIndex(type, version, index);
        }
    }

    void TCaptchaYdbItemsStorage::LoadIndex(TStringBuf type, TCaptchaItemsIndex& index) {
        index.clear();
        ExtendIndex(type, CurrentVersion(type), index);
    }

    TString TCaptchaYdbItemsStorage::CurrentVersion(TStringBuf type) {
        TString strType(type);

        auto paramBuilder = [strType](NYdb::TParamsBuilder&& builder) {
            builder.AddParam("$type").String(strType).Build();
            return builder.Build();
        };

        auto qresult = Client.RunQuery(ESignals::KikimrQueryLoadSingleItemTypeVersionTimingsMs, CurrentVersionSingleQuery, paramBuilder, LoadVersionsTimeout).GetValue(LoadVersionsTimeout);
        Y_ENSURE(qresult.GetResultSets().size() == 1);
        auto parser = qresult.GetResultSetParser(0);

        Y_ENSURE(parser.TryNextRow());
        TString version = parser.ColumnParser("version").GetOptionalString().GetRef();
        Y_ENSURE(!parser.TryNextRow());

        return version;
    }

    void TCaptchaYdbItemsStorage::ExtendIndex(TStringBuf type, TStringBuf version, TCaptchaItemsIndex& index) {
        auto& client = Client.GetClient();

        auto key = NYdb::TValueBuilder()
                       .BeginTuple()
                       .AddElement()
                       .OptionalString(TString(version))
                       .AddElement()
                       .OptionalString(TString(type))
                       .EndTuple()
                       .Build();
        auto border = NYdb::NTable::TKeyBound::Inclusive(key);

        auto readSettings = NYdb::NTable::TReadTableSettings()
                                .From(border)
                                .To(border)
                                .AppendColumns("id")
                                .AppendColumns("metadata")
                                .Ordered();

        auto session = client.GetSession().GetValue(ReadItemIndexTimeout).GetSession();
        auto iter = session.ReadTable(ItemsTableFullPath, readSettings).GetValue(ReadItemIndexTimeout);
        while (true) {
            auto part = iter.ReadNext().GetValue(ReadItemIndexTimeout);

            if (part.EOS()) {
                break;
            }
            if (!part.IsSuccess()) {
                ythrow yexception() << "YDB Error while loading item index: " << part.GetStatus() << " " << part.GetIssues().ToString();
            }

            auto resultSet = part.GetPart();

            NYdb::TResultSetParser parser(resultSet);
            while (parser.TryNextRow()) {
                TCaptchaItemHeader header;
                header.Key.Version = version;
                header.Key.Type = type;
                header.Key.Id = parser.ColumnParser("id").GetOptionalString().GetRef();

                ReadJsonTree(parser.ColumnParser("metadata").GetOptionalString().GetRef(), &header.Metadata, true);
                index.push_back(header);
            }
        }
    }
}
