#include "ydb_session_storage.h"

#include <ydb/public/lib/yson_value/ydb_yson_value.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <util/generic/guid.h>
#include <util/generic/xrange.h>
#include <util/random/random.h>

namespace NCaptchaServer {
    char TCaptchaYdbSessionStorage::YdbTokenTag = '0';

    TCaptchaYdbSessionStorage::TCaptchaYdbSessionStorage(const TCaptchaConfig& config, TCaptchaStats& stats, TCaptchaSessionFactory& sessionFactory)
        : Config(config)
        , Stats(stats)
        , SessionFactory(sessionFactory)
        , CleanerThread(CleanerLoop, this)
    {
        for (auto i : xrange(Config.LocationsSize())) {
            const auto& location = Config.GetLocations(i);

            const auto& idStr = location.GetId();
            Y_VERIFY(idStr.length() == 1);

            char id = idStr[0];
            IdToLocation[id] = location.GetName();
            LocationToId[location.GetName()] = id;

            Connections[id] = MakeHolder<TCaptchaYdbClient>(location.GetDatabase(), Stats);
            SessionTables[id] = location.GetSessionsTable();
            SessionTablesFullPath[id] = location.GetDatabase().GetName() + "/" + location.GetSessionsTable();
        }

        CurrentLocationId = LocationToId.at(Config.GetInstanceLocation());

        QueryTimeout = TDuration::MilliSeconds(Config.GetTimeouts().GetSessionsClientTimeoutMs());
        CleanSessionsQueryTimeout = TDuration::MilliSeconds(Config.GetTimeouts().GetCleanSessionsQueryMs());

        PrepareQueries();

        Stats.RegisterSignalCallback(ESignals::KikimrSessionClientYdbSessions, [this]() { return Connections[CurrentLocationId]->GetClient().GetActiveSessionCount(); });

        CleanerThread.Start();
    }

    NThreading::TFuture<TString> TCaptchaYdbSessionStorage::CreateSession(const TCaptchaSessionRequest& request, TCaptchaSessionInfo& info) {
        SessionFactory.MakeSession(request, ECaptchaItemsStorageId::Kikimr, info);

        TDefaultRng rng;
        const auto& token = BuildToken(&rng, YdbTokenTag, TStringBuf(&CurrentLocationId, 1));

        auto paramBuilder = [token, info](NYdb::TParamsBuilder&& builder) {
            NJsonWriter::TBuf metadata(NJsonWriter::HEM_UNSAFE);
            metadata.WriteJsonValue(&info.Metadata);

            builder.AddParam("$token").String(token).Build();
            builder.AddParam("$type").String(info.Type).Build();
            builder.AddParam("$metadata").String(metadata.Str()).Build();
            builder.AddParam("$timestamp").Uint64(info.Timestamp.MilliSeconds()).Build();
            builder.AddParam("$checks").Int32(info.Checks).Build();
            return builder.Build();
        };

        auto fresult = Connections[CurrentLocationId]->RunQuery(ESignals::KikimrQueryCreateSessionTimingsMs, CreateSessionQuery, paramBuilder, QueryTimeout);
        return fresult.Apply([token](const NYdb::NTable::TAsyncDataQueryResult& fqresult) {
            fqresult.GetValue();
            return token;
        });
    }

    NThreading::TFuture<bool> TCaptchaYdbSessionStorage::LoadSessionInfo(TStringBuf token, TCaptchaSessionInfo& result) {
        TMaybe<char> locationId = LocationFromToken(token);
        if (!locationId.Defined()) {
            return NThreading::MakeFuture(false);
        }

        TString tokenStr(token);

        auto paramBuilder = [tokenStr](NYdb::TParamsBuilder&& builder) {
            builder.AddParam("$token").String(tokenStr).Build();
            return builder.Build();
        };

        auto fresult = Connections[*locationId]->RunQuery(ESignals::KikimrQueryLoadSessionInfoTimingsMs, LoadSessionInfoQueries[*locationId], paramBuilder, QueryTimeout);
        return fresult.Apply([&result](const NYdb::NTable::TAsyncDataQueryResult& fqresult) {
            auto qresult = fqresult.GetValue();
            Y_ENSURE(qresult.GetResultSets().size() == 1);

            auto parser = qresult.GetResultSetParser(0);

            if (parser.TryNextRow()) {
                Y_ENSURE(!parser.TryNextRow());

                result.Type = parser.ColumnParser("type").GetOptionalString().GetRef();
                result.Timestamp = TInstant::MilliSeconds(parser.ColumnParser("timestamp").GetOptionalUint64().GetRef());
                result.Checks = parser.ColumnParser("checks").GetOptionalInt32().GetRef();
                ReadJsonTree(parser.ColumnParser("metadata").GetOptionalString().GetRef(), &result.Metadata, true);
                result.ItemsStorage = ECaptchaItemsStorageId::Kikimr;

                return true;
            }

            return false;
        });
    }

    NThreading::TFuture<bool> TCaptchaYdbSessionStorage::StoreSessionInfo(TStringBuf token, const TCaptchaSessionInfo& info) {
        Y_ASSERT(info.ItemsStorage == ECaptchaItemsStorageId::Kikimr);
        TMaybe<char> locationId = LocationFromToken(token);
        if (!locationId.Defined()) {
            return NThreading::MakeFuture(false);
        }

        TString tokenStr(token);

        auto paramBuilder = [tokenStr, info](NYdb::TParamsBuilder&& builder) {
            NJsonWriter::TBuf metadata(NJsonWriter::HEM_UNSAFE);
            metadata.WriteJsonValue(&info.Metadata);

            builder.AddParam("$token").String(tokenStr).Build();
            builder.AddParam("$checks").Int32(info.Checks).Build();
            builder.AddParam("$metadata").String(metadata.Str()).Build();
            return builder.Build();
        };

        auto fresult = Connections[*locationId]->RunQuery(ESignals::KikimrQueryStoreSessionInfoTimingsMs, StoreSessionInfoQueries[*locationId], paramBuilder, QueryTimeout);
        return fresult.Apply([](const NYdb::NTable::TAsyncDataQueryResult& fqresult) {
            fqresult.GetValue();
            return true;
        });
    }

    NThreading::TFuture<void> TCaptchaYdbSessionStorage::DropSession(TStringBuf token) {
        TMaybe<char> locationId = LocationFromToken(token);
        if (!locationId.Defined()) {
            return NThreading::MakeFuture();
        }

        TString tokenStr(token);

        auto paramBuilder = [tokenStr](NYdb::TParamsBuilder&& builder) {
            builder.AddParam("$token").String(tokenStr).Build();
            return builder.Build();
        };

        auto fresult = Connections[*locationId]->RunQuery(ESignals::KikimrQueryDropSessionTimingsMs, DropSessionQueries[*locationId], paramBuilder, QueryTimeout);
        return fresult.Apply([](const NYdb::NTable::TAsyncDataQueryResult& fqresult) {
            fqresult.GetValue();
        });
    }

    TCaptchaYdbSessionStorage::~TCaptchaYdbSessionStorage() {
        Stopped = true;
        StopEvent.Signal();
        CleanerThread.Join();
    }

    void TCaptchaYdbSessionStorage::PrepareQueries() {
        const char* currentVersionSingleQueryTemplate = R"___(
            -- Create captcha session
            declare $token as String;
            declare $type as String;
            declare $metadata as String;
            declare $timestamp as Uint64;
            declare $checks as Int32;
            upsert into [%s] (token, type, metadata, timestamp, checks) values ($token, $type, $metadata, $timestamp, $checks);
            )___";
        CreateSessionQuery = Sprintf(currentVersionSingleQueryTemplate, SessionTables.at(CurrentLocationId).c_str());

        for (auto& locationClient : Connections) {
            auto locId = locationClient.first;
            const auto& sessTable = SessionTables.at(locId);

            const char* loadSessionInfoQueryTemplate = R"___(
                -- Load captcha session info
                declare $token as String;
                select type, metadata, timestamp, checks from [%s] where token = $token
            )___";
            LoadSessionInfoQueries[locId] = Sprintf(loadSessionInfoQueryTemplate, sessTable.c_str());

            const char* storeSessionInfoQueryTemplate = R"___(
                -- Store captcha session info
                declare $token as String;
                declare $metadata as String;
                declare $checks as Int32;
                upsert into [%s] (token, metadata, checks) values ($token, $metadata, $checks);
            )___";
            StoreSessionInfoQueries[locId] = Sprintf(storeSessionInfoQueryTemplate, sessTable.c_str());

            const char* dropSessionQueryTemplate = R"___(
                -- Drop captcha session
                declare $token as String;
                delete from [%s] where token == $token;
            )___";
            DropSessionQueries[locId] = Sprintf(dropSessionQueryTemplate, sessTable.c_str());
        }

        const char* cleanSessionsQueryTemplate = R"___(
            -- Clean captcha sessions
            declare $tokens as 'List<Struct<token:String>>';

            $tokensSource = (select Item.token as token from (select $tokens as List) flatten by List as Item);

            delete from [%s] on
            select * from $tokensSource;
        )___";
        CleanSessionsQuery = Sprintf(cleanSessionsQueryTemplate, SessionTables.at(CurrentLocationId).c_str());
    }

    TMaybe<char> TCaptchaYdbSessionStorage::LocationFromToken(TStringBuf token) const {
        if (token.length() < 3) {
            DEBUG_LOG << "Token " << TString{token}.Quote() << " is too short" << Endl;
            return Nothing();
        }
        if (!IdToLocation.contains(token[2])) {
            DEBUG_LOG << "Token " << TString{token}.Quote() << " does not correspond to any location" << Endl;
            return Nothing();
        }
        return token[2];
    }

    TVector<NYdb::NTable::TKeyRange> TCaptchaYdbSessionStorage::GetKeyRanges() {
        auto& client = Connections[CurrentLocationId]->GetClient();

        TMaybe<NYdb::NTable::TTableDescription> tableDesc;
        auto status = client.RetryOperationSync([this, &tableDesc](NYdb::NTable::TSession session) {
            auto path = SessionTablesFullPath[CurrentLocationId];
            auto settings = NYdb::NTable::TDescribeTableSettings().WithKeyShardBoundary(true);
            auto result = session.DescribeTable(path, settings).GetValueSync();

            if (result.IsSuccess()) {
                tableDesc.ConstructInPlace(result.GetTableDescription());
            }

            return result;
        });

        if (!status.IsSuccess()) {
            ythrow yexception() << "YDB Error while getting key ranges: " << status.GetStatus() << " " << status.GetIssues().ToString();
        }

        Y_ENSURE(tableDesc);

        return tableDesc->GetKeyRanges();
    }

    void TCaptchaYdbSessionStorage::ReadExpiredTokensChunk(ui64 minTimestamp, TDeque<TString>& tokens, const NYdb::NTable::TKeyRange& keyRange) {
        auto& client = Connections[CurrentLocationId]->GetClient();
        auto timeout = TDuration::MilliSeconds(Config.GetTimeouts().GetReadExpiredSessionsMs());

        auto readSettings = NYdb::NTable::TReadTableSettings()
                                .From(keyRange.From())
                                .To(keyRange.To())
                                .AppendColumns("token")
                                .AppendColumns("timestamp")
                                .RowLimit(Config.GetSessionCleanupReadChunkLimit());

        auto session = client.GetSession().GetValue(timeout).GetSession();
        auto iter = session.ReadTable(SessionTablesFullPath[CurrentLocationId], readSettings).GetValue(timeout);
        while (true) {
            auto part = iter.ReadNext().GetValue(timeout);

            if (part.EOS()) {
                break;
            }
            if (!part.IsSuccess()) {
                ythrow yexception() << "YDB Error while loading expired tokens: " << part.GetStatus() << " " << part.GetIssues().ToString();
            }

            auto resultSet = part.GetPart();

            NYdb::TResultSetParser parser(resultSet);
            while (parser.TryNextRow()) {
                if (parser.ColumnParser("timestamp").GetOptionalUint64().GetOrElse(0) < minTimestamp) {
                    tokens.push_back(parser.ColumnParser("token").GetOptionalString().GetRef());
                }
            }
        }
    }

    static TString KeyRangeToString(const NYdb::NTable::TKeyRange& keyRange) {
        TString result;
        TStringOutput so(result);

        if (!keyRange.From().Defined()) {
            so << "(unbounded";
        } else {
            so << (keyRange.From()->IsInclusive() ? '[' : '(');
            so << NYdb::FormatValueYson(keyRange.From()->GetValue());
        }
        so << ", ";
        if (!keyRange.To().Defined()) {
            so << "unbounded)";
        } else {
            so << NYdb::FormatValueYson(keyRange.To()->GetValue());
            so << (keyRange.To()->IsInclusive() ? ']' : ')');
        }

        return result;
    }

    bool TCaptchaYdbSessionStorage::CleanupIteration(ui64 minTimestamp) {
        bool result = false;
        DEBUG_LOG << "Grabbing tokens for removing..." << Endl;
        for (const auto& keyRange : GetKeyRanges()) {
            TDeque<TString> tokens;

            DEBUG_LOG << "Grabbing tokens for removing in range " << KeyRangeToString(keyRange) << "..." << Endl;
            ReadExpiredTokensChunk(minTimestamp, tokens, keyRange);

            if (tokens.empty()) {
                continue;
            }

            result = true;
            DEBUG_LOG << "Found " << tokens.size() << " tokens, removing..." << Endl;

            size_t length = Config.GetSessionCleanupDeleteChunkLimit();
            size_t begin = 0;
            size_t end = length;
            while (begin < tokens.size()) {
                if (Stopped) {
                    INFO_LOG << "Cleaning up interrupted by exit" << Endl;
                    return false;
                }

                if (end > tokens.size()) {
                    end = tokens.size();
                }

                DEBUG_LOG << "Removing " << (end - begin) << " tokens..." << Endl;

                auto paramBuilder = [begin, end, &tokens](NYdb::TParamsBuilder&& builder) {
                    auto& list = builder.AddParam("$tokens").BeginList();
                    for (auto i : xrange(begin, end)) {
                        list.AddListItem()
                            .BeginStruct()
                            .AddMember("token")
                            .String(tokens[i])
                            .EndStruct();
                    }
                    list.EndList().Build();

                    return builder.Build();
                };

                auto qresult = Connections[CurrentLocationId]->RunQuery(ESignals::KikimrQueryCleanSessionsTimingsMs, CleanSessionsQuery, paramBuilder, CleanSessionsQueryTimeout).GetValue(CleanSessionsQueryTimeout);
                Stats.PushSignal(ESignals::ExpiredSessionsRemoved, end - begin);

                begin += length;
                end += length;
            }
        }

        return result;
    }

    void TCaptchaYdbSessionStorage::Cleanup() {
        ui64 minTimestamp = (Now() - TDuration::Seconds(Config.GetSessionTimeoutSeconds())).MilliSeconds();
        INFO_LOG << "Cleaning up old sessions with timestamp older than " << minTimestamp << Endl;

        while (CleanupIteration(minTimestamp)) {
            if (Stopped) {
                INFO_LOG << "Cleaning up interrupted by exit" << Endl;
                return;
            }
            DEBUG_LOG << "Cleaned up some chunks, trying again" << Endl;
        }

        INFO_LOG << "Cleaned up old sessions" << Endl;
    }

    void* TCaptchaYdbSessionStorage::CleanerLoop(void* ptr) {
        TCaptchaYdbSessionStorage* thisptr = reinterpret_cast<TCaptchaYdbSessionStorage*>(ptr);
        while (true) {
            TDuration cleanupInterval = TDuration::Seconds(thisptr->Config.GetSessionCleanupIntervalSeconds());
            if (thisptr->StopEvent.WaitT(cleanupInterval)) {
                return nullptr;
            }

            try {
                thisptr->Cleanup();
            } catch (std::exception& ex) {
                ERROR_LOG << "Error in session cleanup loop: " << ex.what() << Endl;
            } catch (...) {
                ERROR_LOG << "Error in session cleanup loop (unknown type)" << Endl;
            }
        }
    }
}
