#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/generic/deque.h>
#include <util/generic/hash.h>
#include <util/generic/xrange.h>
#include <util/stream/file.h>
#include <util/string/strip.h>
#include <util/string/printf.h>

struct TOptions {
    TString Endpoint;
    TString Database;
    TString TableItems;
    TString TableVersions;
    TString AuthTokenFile;
    ui64 GracePeriod;
    size_t DeleteChunkSize;
};

struct TCaptchaVersionInfo {
    TString Version;
    ui64 Timestamp;

    TCaptchaVersionInfo() {
    }
    TCaptchaVersionInfo(TString version, ui64 timestamp)
        : Version(version)
        , Timestamp(timestamp)
    {
    }

    bool operator<(const TCaptchaVersionInfo& rhs) const {
        return Timestamp < rhs.Timestamp;
    }

    bool operator==(const TCaptchaVersionInfo& rhs) const {
        return Timestamp == rhs.Timestamp;
    }
};

using TCaptchaVersions = THashMap<TString, TDeque<TCaptchaVersionInfo>>;

TOptions ParseArguments(int argc, char** argv) {
    TOptions result;
    auto opts = NLastGetopt::TOpts::Default();

    opts.AddLongOption("endpoint")
        .StoreResult(&result.Endpoint)
        .Required()
        .Help("Database endpoint");

    opts.AddLongOption("database")
        .StoreResult(&result.Database)
        .Required()
        .Help("Database name");

    opts.AddLongOption("table-items")
        .StoreResult(&result.TableItems)
        .Required()
        .Help("Items table name");

    opts.AddLongOption("table-versions")
        .StoreResult(&result.TableVersions)
        .Required()
        .Help("Versions table name");

    opts.AddLongOption("auth-token-file")
        .StoreResult(&result.AuthTokenFile)
        .DefaultValue("")
        .Help("Path to file with auth token.");

    opts.AddLongOption("grace-period")
        .StoreResult(&result.GracePeriod)
        .DefaultValue(12 * 60 * 60)
        .Help("Only images older by this value than newest version of their type will be removed. Measured in seconds.");

    opts.AddLongOption("delete-chunk-size")
        .StoreResult(&result.DeleteChunkSize)
        .DefaultValue(1000)
        .Help("Delete at most this amount of items in a single query");

    opts.SetFreeArgsNum(0);
    NLastGetopt::TOptsParseResult args{&opts, argc, argv};
    return result;
}

void LoadVersions(const TOptions& options, NYdb::NTable::TTableClient& client, TCaptchaVersions& versions) {
    TMaybe<NYdb::TResultSet> resultSet;
    auto status = client.RetryOperationSync([&resultSet, &options](NYdb::NTable::TSession session) {
        auto query = Sprintf("select type, version, timestamp from [%s]", options.TableVersions.c_str());
        auto result = session.ExecuteDataQuery(query, NYdb::NTable::TTxControl::BeginTx().CommitTx()).GetValueSync();
        if (result.IsSuccess()) {
            resultSet = result.GetResultSet(0);
        }

        return result;
    });

    if (!status.IsSuccess()) {
        ythrow yexception() << "YDB Error: " << status.GetStatus() << " " << status.GetIssues().ToString();
    }

    NYdb::TResultSetParser parser(*resultSet);
    while (parser.TryNextRow()) {
        TString type = parser.ColumnParser("type").GetOptionalString().GetRef();
        TString version = parser.ColumnParser("version").GetOptionalString().GetRef();
        ui64 timestamp = parser.ColumnParser("timestamp").GetOptionalInt32().GetRef();
        versions[type].push_back(TCaptchaVersionInfo(version, timestamp));
    }

    for (auto& typeiter : versions) {
        std::sort(typeiter.second.begin(), typeiter.second.end());
    }
}

void DebugPrintVersions(const TCaptchaVersions& versions) {
    for (const auto& typeiter : versions) {
        Cerr << "Type: " << typeiter.first.Quote() << Endl;
        for (const auto& verinfo : typeiter.second) {
            Cerr << "    " << verinfo.Version.Quote() << ": " << verinfo.Timestamp << Endl;
        }
    }
}

void SelectVersionsToRemove(ui64 gracePeriod, const TCaptchaVersions& versions, TCaptchaVersions& result) {
    for (const auto& iter : versions) {
        size_t size = iter.second.size();
        Y_ASSERT(size > 0);

        ui64 newestTimestamp = iter.second[size - 1].Timestamp;

        //always keep last 2 versions
        if (size <= 2) {
            continue;
        }
        for (auto i : xrange(size - 2)) {
            ui64 timestamp = iter.second[i].Timestamp;
            Y_ASSERT(newestTimestamp > timestamp);
            if (newestTimestamp - timestamp > gracePeriod) {
                result[iter.first].push_back(iter.second[i]);
            }
        }
    }
}

void LoadItemIds(const TOptions& options, NYdb::NTable::TTableClient& client, TStringBuf type, TStringBuf version, TDeque<TString>& result) {
    auto key = NYdb::TValueBuilder()
                   .BeginTuple()
                   .AddElement()
                   .OptionalString(TString(version))
                   .AddElement()
                   .OptionalString(TString(type))
                   .EndTuple()
                   .Build();
    auto border = NYdb::NTable::TKeyBound::Inclusive(key);

    auto readSettings = NYdb::NTable::TReadTableSettings()
                            .From(border)
                            .To(border)
                            .AppendColumns("id")
                            .Ordered();

    auto session = client.GetSession().GetValueSync().GetSession();
    auto iter = session.ReadTable(options.Database + "/" + options.TableItems, readSettings).GetValueSync();
    while (true) {
        auto part = iter.ReadNext().GetValueSync();

        if (part.EOS()) {
            break;
        }
        if (!part.IsSuccess()) {
            ythrow yexception() << "YDB Error while loading item ids: " << part.GetStatus() << " " << part.GetIssues().ToString();
        }

        auto resultSet = part.GetPart();

        NYdb::TResultSetParser parser(resultSet);
        while (parser.TryNextRow()) {
            result.push_back(parser.ColumnParser("id").GetOptionalString().GetRef());
        }
    }
}

void RemoveItems(const TOptions& options, NYdb::NTable::TTableClient& client, TStringBuf type, TStringBuf version) {
    TDeque<TString> ids;
    LoadItemIds(options, client, type, version, ids);

    size_t length = options.DeleteChunkSize;
    size_t begin = 0;
    size_t end = length;
    while (begin < ids.size()) {
        if (end > ids.size()) {
            end = ids.size();
        }

        auto paramBuilder = [type, version, begin, end, &ids](NYdb::TParamsBuilder&& builder) {
            builder.AddParam("$type").String(TString(type)).Build();
            builder.AddParam("$version").String(TString(version)).Build();
            auto& list = builder.AddParam("$ids").BeginList();
            for (auto i : xrange(begin, end)) {
                list.AddListItem()
                    .BeginStruct()
                    .AddMember("id")
                    .String(ids[i])
                    .EndStruct();
            }
            list.EndList().Build();

            return builder.Build();
        };

        auto status = client.RetryOperationSync([&options, &paramBuilder](NYdb::NTable::TSession session) {
            const char* deleteItemsQueryTemplate = R"___(
                declare $type as String;
                declare $version as String;
                declare $ids as 'List<Struct<id:String>>';

                $idsSource = (select $type as type, $version as version, Item.id as id from (select $ids as List) flatten by List as Item);

                delete from [%s] on
                select * from $idsSource;
            )___";
            auto deleteItemsQuery = Sprintf(deleteItemsQueryTemplate, options.TableItems.c_str());

            return session.ExecuteDataQuery(deleteItemsQuery, NYdb::NTable::TTxControl::BeginTx().CommitTx(), paramBuilder(session.GetParamsBuilder())).GetValueSync();
        });

        if (!status.IsSuccess()) {
            ythrow yexception() << "YDB Error: " << status.GetStatus() << " " << status.GetIssues().ToString();
        }

        Cerr << "Removed " << end << "/" << ids.size() << " items" << Endl;

        begin += length;
        end += length;
    }

    auto paramBuilder = [type, version](NYdb::TParamsBuilder&& builder) {
        builder.AddParam("$type").String(TString(type)).Build();
        builder.AddParam("$version").String(TString(version)).Build();
        return builder.Build();
    };

    auto status = client.RetryOperationSync([&options, &paramBuilder](NYdb::NTable::TSession session) {
        const char* deleteVersionQueryTemplate = R"___(
            declare $type as String;
            declare $version as String;
            delete from [%s] where type = $type and version = $version;
            )___";
        auto deleteVersionQuery = Sprintf(deleteVersionQueryTemplate, options.TableVersions.c_str());

        return session.ExecuteDataQuery(deleteVersionQuery, NYdb::NTable::TTxControl::BeginTx().CommitTx(), paramBuilder(session.GetParamsBuilder())).GetValueSync();
    });

    if (!status.IsSuccess()) {
        ythrow yexception() << "YDB Error: " << status.GetStatus() << " " << status.GetIssues().ToString();
    }

    Cerr << "Removed version record" << Endl;
}

int main(int argc, char** argv) {
    TOptions options = ParseArguments(argc, argv);

    auto driverConfig = NYdb::TDriverConfig()
                            .SetEndpoint(options.Endpoint)
                            .SetDatabase(options.Database);

    if (options.AuthTokenFile) {
        TString token = Strip(TFileInput(options.AuthTokenFile).ReadAll());
        driverConfig.SetAuthToken(token);
    }

    auto driver = NYdb::TDriver(driverConfig);
    auto client = NYdb::NTable::TTableClient(driver);

    TCaptchaVersions versions;
    LoadVersions(options, client, versions);

    TCaptchaVersions versionsToRemove;
    SelectVersionsToRemove(options.GracePeriod, versions, versionsToRemove);

    for (const auto& typeiter : versionsToRemove) {
        for (const auto& verinfo : typeiter.second) {
            Cerr << "Removing items with type=" << typeiter.first.Quote() << " version=" << verinfo.Version.Quote() << "..." << Endl;
            RemoveItems(options, client, typeiter.first, verinfo.Version);
        }
    }
}
