#!/usr/bin/env python

import argparse

import ydb


TOKEN_ALPHABET = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'


def parse_args():
    parser = argparse.ArgumentParser(description='Make sharded sessions table')
    parser.add_argument('-E', '--endpoint', required=True, help='YDB endpoint')
    parser.add_argument('-D', '--database', required=True, help='Database name')
    parser.add_argument('-T', '--table', required=True, help='Session table name')
    parser.add_argument('-P', '--partition-count', default=100, type=int, help='Partition count')
    parser.add_argument('-A', '--auth-token-file', type=argparse.FileType('r'), help='File with auth token')
    parser.add_argument('--token-prefix', default='00A', help='Session token prefix')
    parser.add_argument('--token-length', default=32, type=int, help='Token total length (including the prefix)')

    return parser.parse_args()


def longint_to_string(val, alphabet, minlength=1):
    result = ''
    while val > 0:
        result = alphabet[val % len(alphabet)] + result
        val /= len(alphabet)

    result = result.rjust(minlength, alphabet[0])
    return result


def build_boundaries(token_prefix, token_length, count, alphabet=TOKEN_ALPHABET):
    alphabet = ''.join(sorted(alphabet))
    random_part_length = token_length - len(token_prefix)
    if random_part_length <= 0:
        raise ValueError('Token length is too small')

    random_part_max = len(alphabet)**random_part_length - 1
    partition_size = random_part_max / count
    result = []
    for i in xrange(1, count):
        boundary_int = partition_size * i
        boundary = token_prefix + longint_to_string(boundary_int, alphabet, random_part_length)
        assert len(boundary) == token_length
        result.append(boundary)
    return result


def create_session_table(session, table, boundaries):
    partition_spec = []
    for boundary in boundaries:
        partition_spec.append(ydb.KeyBound((boundary,)))

    description = (
        ydb.TableDescription()
        .with_primary_key('token')
        .with_columns(
            ydb.Column('token', ydb.OptionalType(ydb.PrimitiveType.String)),
            ydb.Column('type', ydb.OptionalType(ydb.PrimitiveType.String)),
            ydb.Column('metadata', ydb.OptionalType(ydb.PrimitiveType.String)),
            ydb.Column('checks', ydb.OptionalType(ydb.PrimitiveType.Int32)),
            ydb.Column('timestamp', ydb.OptionalType(ydb.PrimitiveType.Uint64))
        )
        .with_profile(
            ydb.TableProfile()
            .with_partitioning_policy(
                ydb.PartitioningPolicy()
                .with_explicit_partitions(
                    ydb.ExplicitPartitions(
                        partition_spec
                    )
                )
            )
        )
    )
    session.create_table(table, description)


def main():
    args = parse_args()
    auth_token = None
    if args.auth_token_file:
        auth_token = args.auth_token_file.read().strip()

    boundaries = build_boundaries(args.token_prefix, args.token_length, args.partition_count)
    print 'Boundaries:'
    for boundary in boundaries:
        print ' - %s' % boundary

    connection_params = ydb.DriverConfig(args.endpoint, args.database, auth_token=auth_token)
    driver = ydb.Driver(connection_params)
    try:
        driver.wait(timeout=5)
        session = driver.table_client.session().create()
        create_session_table(session, args.database + '/' + args.table, boundaries)
    finally:
        driver.stop()

if __name__ == '__main__':
    main()
