#!/usr/bin/env python

import os
import StringIO
import logging
import argparse
from PIL import Image
from captcha.generation.image_generator.noise import whitenoise


LOGGING_FMT = u'%(asctime)s | %(levelname)-.1s | [%(process)d:%(threadName)s:%(thread)X] '\
              u'| %(name)s [%(filename)s:%(lineno)d] | %(message)s'

logging.basicConfig(format=LOGGING_FMT)
_LOG = logging.getLogger(__name__)
_LOG.setLevel(logging.INFO)


def chop_header(egif_file):
    egif = open(egif_file, 'rb').read()
    header_pos = egif.find('GIF89a')
    if header_pos == -1:
        header_pos = egif.find('GIF87a')
    if header_pos == -1:
        raise ValueError('bad image ' + egif_file)
    clean_img = egif[header_pos:]
    return egif[:header_pos], Image.open(StringIO.StringIO(clean_img))


def add_header(img, header, egif_file):
    with open(egif_file, 'wb') as egif:
        output = StringIO.StringIO()
        img.save(output, format="GIF")
        egif.write(header)
        egif.write(output.getvalue())


def process_single_image(src_img, dst_img, **kwargs):
    keep_header = kwargs.pop('keep_header', False)
    header, img = chop_header(src_img)
    noisy_img = whitenoise.add_noise_to_single_img(img, **kwargs)
    if keep_header:
        add_header(noisy_img, header, dst_img)
    else:
        noisy_img.save(dst_img, format="GIF")
    _LOG.info('done: {img}'.format(img=dst_img))


def process_img_dir(src_dir, dst_dir, **kwargs):
    if not os.path.isdir(dst_dir):
        os.makedirs(dst_dir)
    for loc in os.listdir(src_dir):
        src_loc = os.path.join(src_dir, loc)
        dst_loc = os.path.join(dst_dir, loc)
        if os.path.isdir(src_loc):
            process_img_dir(src_loc, dst_loc, **kwargs)
        elif os.path.isfile(src_loc):
            process_single_image(src_loc, dst_loc, **kwargs)


def main():
    argparser = argparse.ArgumentParser()
    argparser.add_argument('-s', dest='src', required=True)
    argparser.add_argument('-d', dest='dst', required=True)
    argparser.add_argument('--keep_header', action='store_true')
    argparser.add_argument('--scale', type=float, default=whitenoise.DEFAULT_SCALE)
    argparser.add_argument('--single_image', action='store_true')
    args = argparser.parse_args()

    call_func = process_single_image if args.single_image else process_img_dir
    call_func(args.src, args.dst, scale=args.scale, keep_header=args.keep_header)


if __name__ == '__main__':
    main()
