package ru.yandex.ci.engine.config;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.ci.core.config.ConfigProblem;
import ru.yandex.ci.core.config.ConfigUtils;
import ru.yandex.ci.core.config.a.model.AYamlConfig;
import ru.yandex.ci.core.config.a.model.CiConfig;
import ru.yandex.ci.core.config.a.model.FlowConfig;
import ru.yandex.ci.core.config.a.model.FlowWithFlowVars;
import ru.yandex.ci.core.config.a.model.JobConfig;
import ru.yandex.ci.core.config.a.model.JobSkippedByMode;
import ru.yandex.ci.core.config.a.model.ReleaseConfig;
import ru.yandex.ci.core.config.a.model.StageConfig;
import ru.yandex.ci.core.config.registry.RollbackMode;
import ru.yandex.ci.core.config.registry.TaskConfig;
import ru.yandex.ci.core.config.registry.TaskId;

@Slf4j
@AllArgsConstructor
public class ConfigParsePostProcessor {

    @Nonnull
    private final AYamlConfig originalConfig;

    @Nonnull
    private final NavigableMap<TaskId, TaskConfig> taskConfigs;

    @Nonnull
    private final List<ConfigProblem> problems;

    AYamlConfig postProcess() {
        return new AutoRollbacks().prepareAutoRollbackFlow();
    }

    private class AutoRollbacks {

        private AYamlConfig prepareAutoRollbackFlow() {
            var config = originalConfig;
            for (ReleaseConfig release : config.getCi().getReleases().values()) {
                if (hasRollbacks(release)) {
                    log.info("Trying to generate rollback flow for release {}", release.getId());
                    config = maybeAddAutoRollback(config, release);
                } else {
                    log.debug("Unable to generate rollback flow for release {}: no accepted stages", release.getId());
                    if (!release.getRollbackFlows().isEmpty()) {
                        problems.add(ConfigProblem.crit("Rollbacks",
                                String.format("Invalid rollback configuration for release %s: rollback flows " +
                                                "will not work without at least one stage with 'rollback' attribute",
                                        release.getId())));
                    }
                }
            }
            return config;
        }

        private AYamlConfig maybeAddAutoRollback(AYamlConfig config, ReleaseConfig releaseConfig) {
            var ci = config.getCi();

            var stageSet = releaseConfig.getStages().stream()
                    .filter(StageConfig::isRollback)
                    .map(StageConfig::getId)
                    .collect(Collectors.toSet());

            // Generate rollback for each normal and hotfix flow (if possible)
            var generatedFlows = new ArrayList<AutoGeneratedFlow>(4);
            Consumer<FlowWithFlowVars> addFlow = ref ->
                    buildFlow(ci, ref, releaseConfig, stageSet)
                            .ifPresent(generatedFlows::add);

            addFlow.accept(new FlowWithFlowVars(releaseConfig.getFlow(), releaseConfig.getFlowVars(), null, null));
            releaseConfig.getHotfixFlows().forEach(addFlow);

            if (generatedFlows.isEmpty()) {
                return config;
            }

            var releases = new LinkedHashMap<>(ci.getReleases());
            var rollbackFlows = new ArrayList<>(releaseConfig.getRollbackFlows());
            generatedFlows.forEach(flow -> rollbackFlows.add(flow.getFlowReference()));
            releases.put(releaseConfig.getId(), releaseConfig.withRollbackFlows(rollbackFlows));

            var flows = new LinkedHashMap<>(ci.getFlows());
            generatedFlows.forEach(flow -> flows.put(flow.getFlowConfig().getId(), flow.getFlowConfig()));

            var newCi = ci
                    .withReleases(releases)
                    .withFlows(flows);
            return config.withCi(newCi);
        }

        private Optional<AutoGeneratedFlow> buildFlow(CiConfig ci,
                                                      FlowWithFlowVars sourceRef,
                                                      ReleaseConfig releaseConfig,
                                                      Set<String> acceptStages) {
            var sourceFlowId = sourceRef.getFlow();
            var sourceFlow = ci.getFlow(sourceFlowId);

            var invalidReason = autoRollbackInvalidReason(sourceFlow, acceptStages, taskConfigs);
            if (invalidReason != null) {
                if (!hasManualRollbacks(releaseConfig, sourceRef)) {
                    problems.add(ConfigProblem.crit("Rollbacks",
                            String.format("Unable to generate rollback flow for release %s, flow %s: " +
                                            invalidReason,
                                    releaseConfig.getId(), sourceFlowId)));
                } else {
                    log.info("Unable to generate rollback: {}", invalidReason);
                }
                return Optional.empty(); // ---
            }

            var flowId = "rollback_" + releaseConfig.getId() + "_" + sourceFlowId;

            var flowBuilder = buildAutoRollbackFlow(sourceFlow, acceptStages, taskConfigs);
            flowBuilder.id(flowId);
            flowBuilder.title(sourceFlow.getTitle());

            Preconditions.checkState(ci.findFlow(flowId).isEmpty(),
                    "Attempt to generated rollback flow with duplicate id: %s", flowId);

            log.info("Generated rollback flow: {}", flowId);

            var ref = new FlowWithFlowVars(flowId, sourceRef.getFlowVars(), null, Set.of(sourceFlowId));
            return Optional.of(new AutoGeneratedFlow(flowBuilder.build(), ref));
        }

        private static boolean hasRollbacks(ReleaseConfig release) {
            return release.getStages().stream().anyMatch(StageConfig::isRollback);
        }

        private static boolean hasManualRollbacks(ReleaseConfig releaseConfig, FlowWithFlowVars flowRef) {
            for (FlowWithFlowVars rollbackFlow : releaseConfig.getRollbackFlows()) {
                if (rollbackFlow.acceptFlow(flowRef.getFlow())) {
                    return true;
                }
            }
            return false;
        }

        @Nullable
        private static String autoRollbackInvalidReason(FlowConfig flow,
                                                        Set<String> acceptStages,
                                                        NavigableMap<TaskId, TaskConfig> taskConfigs) {
            var tasks = Stream.of(flow.getJobs(), flow.getCleanupJobs())
                    .map(LinkedHashMap::values)
                    .flatMap(Collection::stream)
                    .filter(job -> acceptStages.contains(job.getStage()))
                    .map(JobConfig::getTaskId)
                    .filter(TaskId::requireTaskConfig)
                    .distinct()
                    .toList();

            if (tasks.isEmpty()) {
                return "no actual tasks to run";
            }
            return tasks.stream()
                    .map(id -> {
                        var cfg = taskConfigs.get(id);
                        if (cfg == null) {
                            return "cannot find registry for task " + id;
                        }
                        if (cfg.getAutoRollbackMode() == RollbackMode.DENY) {
                            return "task " + id + " does not allow rollbacks";
                        }
                        return null;
                    })
                    .filter(Objects::nonNull)
                    .findAny()
                    .orElse(null);
        }

        private static FlowConfig.Builder buildAutoRollbackFlow(FlowConfig flow,
                                                                Set<String> acceptStages,
                                                                NavigableMap<TaskId, TaskConfig> taskConfigs) {
            // Task list: clone flow, filter jobs, register in flow list, register in rollbacks in releases
            // We think the rollback flow looks OK...

            Function<JobConfig, JobConfig> updateJob = job -> {
                var taskId = job.getTaskId();

                if (!acceptStages.contains(job.getStage())) {
                    return job.withSkippedBy(JobSkippedByMode.STAGE);
                }

                if (!taskId.requireTaskConfig()) {
                    return job; // ---
                }

                var task = taskConfigs.get(taskId);
                Preconditions.checkState(task != null, "Unable to find task %s", taskId);
                switch (task.getAutoRollbackMode()) {
                    case SKIP -> {
                        return job.withSkippedBy(JobSkippedByMode.REGISTRY);
                    }
                    case EXECUTE -> {
                        return job; // accept as is
                    }
                    default -> throw new IllegalStateException("Internal error, cannot generate rollback flow: " +
                            task.getAutoRollbackMode());
                }
            };

            Function<LinkedHashMap<String, JobConfig>, LinkedHashMap<String, JobConfig>> updateJobs = jobs ->
                    ConfigUtils.toMap(jobs.values().stream().map(updateJob).toList());

            return flow.toBuilder()
                    .jobs(updateJobs.apply(flow.getJobs()))
                    .cleanupJobs(updateJobs.apply(flow.getCleanupJobs()));
        }

        @Value
        private static class AutoGeneratedFlow {
            @Nonnull
            FlowConfig flowConfig;
            @Nonnull
            FlowWithFlowVars flowReference;
        }
    }
}
