# coding: utf-8
"""xlsx-keywords:
  - table_total_campaigns_stat
  - 00_used_strategies
  - 04_campaigns_type
  - 04_list_campaigns_type

Help:
[Раздельное управление площадками]
  (https://yandex.ru/support/direct/strategies/independent.xml?lang=en)
[Creating a separate campaign for displaying ads in the ad networks]
  (https://yandex.ru/support/direct/efficiency/content-websites.xml?lang=en)
"""
from ..review import Recommendation, append_recommendation
from ..utils import get_all_campaigns, get_cid, get_strategy_type, filter_by_adtype, review_keywords_url
from .utils import (
    add_cpc_search,
    add_cpc_spec,
    add_cpc_rsya,
    add_cpc_other,
    add_cpc_total,
    add_ctr_search,
    add_ctr_spec,
    add_ctr_rsya,
    add_ctr_other,
    add_ctr_total,
)
from ast import literal_eval
import pandas as pd


def translate_strategy_type(row, adv_type=None):
    assert adv_type in {'Search', 'Network', None}

    search_strategies = {
        'HIGHEST_POSITION': u'Ручное управление ставками',
        'IMPRESSIONS_BELOW_SEARCH': u'Ручное управление ставками. Показ под результатами Поиска',
        'WB_MAXIMUM_CLICKS': u'Недельный бюджет. Максимум кликов',
        'WB_MAXIMUM_CONVERSION_RATE': u'Недельный бюджет. Максимальная конверсия',
        'AVERAGE_CPC': u'Средняя цена клика',
        'AVERAGE_CPA': u'Средняя цена конверсии',
        'AVERAGE_ROI': u'Средняя рентабельность инвестиций',
        'WEEKLY_CLICK_PACKAGE': u'Недельный пакет кликов',
        'SERVING_OFF': u'Показы отключены',
        'WB_MAXIMUM_APP_INSTALLS': u'Недельный бюджет. Максимальное количество установок',
        'AVERAGE_CPI': u'Средняя цена установки',
        'UNKNOWN': u'Unknown'}

    network_strategies = {
        'NETWORK_DEFAULT': u'Ручное управление ставками (Поиск и Сети не разделены)',
        'MAXIMUM_COVERAGE': u'Ручное управление ставками. Раздельно управлять ставками на Поиске и в Сетях',
        'WB_MAXIMUM_CLICKS': u'Недельный бюджет. Максимум кликов',
        'WB_MAXIMUM_CONVERSION_RATE': u'Недельный бюджет. Максимальная конверсия',
        'AVERAGE_CPC': u'Средняя цена клика',
        'AVERAGE_CPA': u'Средняя цена конверсии',
        'AVERAGE_ROI': u'Средняя рентабельность инвестиций',
        'WEEKLY_CLICK_PACKAGE': u'Недельный пакет кликов',
        'SERVING_OFF': u'Показы отключены',
        'WB_MAXIMUM_APP_INSTALLS': u'Недельный бюджет. Максимальное количество установок',
        'AVERAGE_CPI': u'Средняя цена установки',
        'UNKNOWN': u'Unknown'}

    if adv_type == 'Search':
        return search_strategies.get(row, row)
    elif adv_type == 'Network':
        return network_strategies.get(row, row)

    has_rsya = row['ShowsRSYA'] > 0
    has_search = row['ShowsSearch'] > 0

    if has_rsya and has_search:
        return u'{search_strategy}; {rsya_strategy}'.format(
            search_strategy=search_strategies.get(row['SearchStrategy'], row['SearchStrategy']),
            rsya_strategy=network_strategies.get(row['RSYAStrategy'], row['RSYAStrategy']))
    elif has_rsya:
        return network_strategies.get(row['RSYAStrategy'], row['RSYAStrategy'])
    elif has_search:
        return search_strategies.get(row['SearchStrategy'], row['SearchStrategy'])
    else:
        return 'None'


def extract_strategy(cell, adv_type):
    """
    Returns value of "BiddingStrategyType" or "None"

    Params:
    cell :: str - encoded dict:
        u"{u'Search': {u'BiddingStrategyType': u'HIGHEST_POSITION'},
            u'Network': {u'BiddingStrategyType': u'SERVING_OFF'}}"
    adv_type :: "Search" | "Network"
    """
    assert adv_type in {"Search", "Network"}
    none_val = "None"

    try:
        strategy_opt = literal_eval(cell)
    except SyntaxError:
        return none_val

    return strategy_opt.get(adv_type, {"BiddingStrategyType": none_val}).get("BiddingStrategyType", none_val)


def table_campaigns_overview(df, groupby="CampaignID", show_total=False):
    """00_campaigns_overview, 00_logins_overview"""

    def overlap_type(overlap):

        if overlap >= .99:
            return u'full'

        if overlap <= .01:
            return u'none'

        return u'partly'

    GROUPBY_TYPES = ["CampaignID", "LoginName"]
    assert groupby in GROUPBY_TYPES, "`groupby` is not in %s" % GROUPBY_TYPES

    df_chart = filter_by_adtype(df, "TEXT_AD")
    if df_chart.empty:
        return None

    cols = ["LoginName", "CampaignID", "CampaignName", "Strategy", "DirectBannerID", "isMobileBanner",
            "BoldedWordsCnt", "WordsCnt",
            "ShowsRSYA", "ShowsSearch", "Shows", "Clicks", "Cost",
            "SitelinksCnt",
            "CalloutsCnt",
            "DisplayUrlHasTemplate",
            "ImageHash"]

    df_chart["StrategyType"] = df_chart.apply(get_strategy_type, axis=1)
    if groupby == "CampaignID":
        group_cols = ["LoginName", "CampaignID", "Strategy"]
    elif groupby == "LoginName":
        group_cols = ["LoginName", "Strategy"]

    camp_df = (df_chart.copy()[["LoginName", "StrategyType", "CampaignID"]]
               .drop_duplicates().dropna()
               .set_index(["LoginName", "StrategyType"]))

    df_chart["PhraseBoldShare"] = df_chart.BoldedWordsCnt / df_chart.WordsCnt
    df_chart["PhraseBoldType"] = df_chart.PhraseBoldShare.apply(overlap_type).astype(str)

    df_chart["FullBoldShows"] = df_chart.Shows.where(df_chart.PhraseBoldType == "full")
    df_chart["PartlyBoldShows"] = df_chart.Shows.where(df_chart.PhraseBoldType == "partly")

    df_chart["ShowsWithoutImgs"] = df_chart.ShowsRSYA.where(df_chart.ImageHash != 0)

    df_chart["DisplayUrlHasTemplate"] = df_chart.DisplayUrlHasTemplate.astype(str)
    df_chart["ShowsWithTemplate"] = df_chart.Shows.where(df_chart.DisplayUrlHasTemplate != "None")

    df_chart["NotMobileShows"] = df_chart.Shows.where(df_chart.isMobileBanner == False)
    df_chart["ShowsWithCallouts"] = df_chart.Shows.where(
        (df_chart.isMobileBanner == False) & (df_chart.CalloutsCnt != 0)
    )
    df_chart["ShowsWithSitelinks"] = df_chart.Shows.where(df_chart.SitelinksCnt != 0)

    df_chart = (df_chart
                .fillna(value=0)
                .groupby(group_cols + ["DateBegin", "DateFinish"], as_index=False)
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "FullBoldShows": pd.np.sum,
                      "PartlyBoldShows": pd.np.sum,
                      "ShowsSearch": pd.np.sum,
                      "ShowsRSYA": pd.np.sum,
                      "ShowsWithSitelinks": pd.np.sum,
                      "ShowsWithCallouts": pd.np.sum,
                      "NotMobileShows": pd.np.sum,
                      "ShowsWithTemplate": pd.np.sum,
                      "ShowsWithoutImgs": pd.np.sum
                      })
                )

    df_chart = (df_chart.groupby(group_cols + ["DateBegin", "DateFinish"], as_index=False).sum()
                .assign(ShowsWithSitelinksShare=lambda df: df.ShowsWithSitelinks / df.Shows,
                        ShowsWithCalloutsShare=lambda df: df.ShowsWithCallouts / df.NotMobileShows,
                        ShowsWithTemplateShare=lambda df: df.ShowsWithTemplate / df.Shows,
                        FullBoldShare=lambda df: df.FullBoldShows / df.Shows,
                        PartlyBoldShare=lambda df: df.PartlyBoldShows / df.Shows,
                        ShowsWithoutImgsShare=lambda df: df.ShowsWithoutImgs / df.ShowsRSYA
                        )
                .fillna(value=0)
                )

    df_chart = (df_chart
                .assign(SearchStrategy=lambda df:
                        df.Strategy.apply(lambda x: translate_strategy_type(extract_strategy(x, "Search"), "Search")),
                        RSYAStrategy=lambda df:
                        df.Strategy.apply(lambda x: translate_strategy_type(extract_strategy(x, "Network"), "Network"))
                        )
                .assign(
                    StrategyType=lambda df: df.apply(lambda row: get_strategy_type(row), axis=1),
                    SearchIsAutoStrateg=lambda df: df.SearchStrategy.apply(lambda x: x.lower().find(u"ручное") == -1),
                    RSYAIsAutoStrateg=lambda df: df.SearchStrategy.apply(lambda x: x.lower().find(u"ручное") == -1))
                .fillna(value=0)
                )

    df_chart["Campaigns"] = df_chart.apply(lambda row: get_all_campaigns(row, camp_df), axis=1)
    df_chart["Url"] = df_chart.apply(lambda row: review_keywords_url(row), axis=1)
    if groupby == "LoginName":
        group_cols.insert(1, "Url")

    if groupby == "CampaignID":
        total_strat_cid = df_chart.shape[0]
        custom_strat_cid = df_chart.query("not (SearchIsAutoStrateg and RSYAIsAutoStrateg)").shape[0]
        if custom_strat_cid > 0:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="00_campaigns_overview",
                title=u"Автоматические стратегии",
                snippet=u"%d кампаний (%.0f%%) управляются вручную"
                        % (custom_strat_cid, custom_strat_cid * 100. / total_strat_cid),
                importance_lvl=2)
            append_recommendation(df, Rec, uniq=False)

        both_strat_cid = df_chart.query("StrategyType == 'both'").shape[0]
        if both_strat_cid > 0:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="00_campaigns_overview",
                title=u"Автоматические стратегии",
                snippet=u"%d кампаний (%.0f%%) не разделены по типу площадки"
                        % (both_strat_cid, both_strat_cid * 100. / total_strat_cid),
                importance_lvl=2)
            append_recommendation(df, Rec, uniq=False)

    df_chart.StrategyType = df_chart.StrategyType.replace(to_replace={
        "only_search": u"Только поиск",
        "only_rsya": u"Только сети",
        "both": u"Все площадки"})

    if show_total:
        df_chart = pd.DataFrame(df_chart.sum(axis=0, numeric_only=True)).T
        for col in ["LoginName", "CampaignID", "Url", "SearchStrategy", "RSYAStrategy", "StrategyType"]:
            df_chart[col] = ""
        df_chart = (df_chart.assign(ShowsWithSitelinksShare=lambda df: df.ShowsWithSitelinks / df.Shows,
                                    ShowsWithCalloutsShare=lambda df: df.ShowsWithCallouts / df.NotMobileShows,
                                    ShowsWithTemplateShare=lambda df: df.ShowsWithTemplate / df.Shows,
                                    FullBoldShare=lambda df: df.FullBoldShows / df.Shows,
                                    PartlyBoldShare=lambda df: df.PartlyBoldShows / df.Shows,
                                    ShowsWithoutImgsShare=lambda df: df.ShowsWithoutImgs / df.ShowsRSYA
                                    )
                    .fillna(value=0)
                    )

    if groupby == "CampaignID":
        cols_order = ["LoginName", "CampaignID", "SearchStrategy", "RSYAStrategy", "StrategyType", "Cost",
                      "FullBoldShare", "PartlyBoldShare",
                      "ShowsWithSitelinksShare",
                      "ShowsWithCalloutsShare",
                      "ShowsWithTemplateShare",
                      "ShowsWithoutImgsShare",
                      ]
    elif groupby == "LoginName":
        cols_order = ["LoginName", "Url", "SearchStrategy", "RSYAStrategy", "StrategyType", "Shows", "Clicks", "Cost",
                      "FullBoldShare", "PartlyBoldShare",
                      "ShowsWithSitelinksShare",
                      "ShowsWithCalloutsShare",
                      "ShowsWithTemplateShare",
                      "ShowsWithoutImgsShare",
                      ]
    df_chart = df_chart[cols_order]

    cols = {"LoginName": u"Логин",
            "CampaignID": u"Кампания",
            "Url": u"Ссылка на кампании",
            "SearchStrategy": u"Cтратегия на поиске",
            "RSYAStrategy": u"Cтратегия в сетях",
            "StrategyType": u"Подключенные площадки",
            "Shows": u"Показы",
            "Clicks": u"Клики",
            "Cost": u"Расходы",
            "FullBoldShare": u"Доля показов с полной подсветкой",
            "PartlyBoldShare": u"Доля показов с частичной подсветкой",
            "ShowsWithSitelinksShare": u"Доля показов с БС",
            "ShowsWithCalloutsShare": u"Доля показов с уточнениями",
            "ShowsWithTemplateShare": u"Доля показов с отображаемой ссылкой",
            "ShowsWithoutImgsShare": u"Доля показов с картинками в сетях",
            }
    for col in df_chart.columns:
        df_chart.rename(columns={col: cols.get(col, col)}, inplace=True)

    setattr(df_chart, "formats", {u"Логин": "string",
                                  u"Cтратегия на поиске": "string",
                                  u"Cтратегия в сетях": "string",
                                  u"Подключенные площадки": "string",
                                  u"Расходы": "float"})
    if groupby == "LoginName":
        df_chart.formats[u"Показы"] = "int"
        df_chart.formats[u"Клики"] = "int"
        df_chart.formats[u"Ссылка на кампании"] = "hyperlink"
    elif groupby == "CampaignID":
        df_chart.formats[u"Кампания"] = "string"

    return df_chart


def table_total_campaigns_stat(df):
    """00_total_campaigns_stat"""

    df_chart = (df[["ShowsSpec", "ClicksSpec", "CostSpec",
                    "ShowsOther", "ClicksOther", "CostOther",
                    "ShowsRSYA", "ClicksRSYA", "CostRSYA",
                    "CampaignID", "CampaignName", "DayBudget", "Strategy"]]
                .groupby(["CampaignID", "CampaignName", "DayBudget", "Strategy"], as_index=False)
                .sum()
                .assign(SearchStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Search"), "Search")))
                .assign(RSYAStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Network"), "Network")))
                .assign(HelperUrl=lambda df: df.CampaignID.apply(get_cid))
                .drop("Strategy", axis=1))
    add_cpc_spec(df_chart)
    add_cpc_other(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_spec(df_chart)
    add_ctr_other(df_chart)
    add_ctr_rsya(df_chart)
    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "DayBudget": u"Дневной бюджет",
        "ShowsSpec": u"Показы (спец.)",
        "ClicksSpec": u"Клики (спец.)",
        "CostSpec": u"Расходы (спец.)",
        "CPCSpec": u"CPC (спец.)",
        "CTRSpec": u"CTR (спец.)",
        "ShowsOther": u"Показы (юж./прав. блок)",
        "ClicksOther": u"Клики (юж./прав. блок)",
        "CostOther": u"Расходы (юж./прав. блок)",
        "CPCOther": u"CPC (юж./прав. блок)",
        "CTROther": u"CTR (юж./прав. блок)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Расходы (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)",
        "SearchStrategy": u"Стратегия на поиске",
        "RSYAStrategy": u"Стратегия в сетях",
        "HelperUrl": u"Ссылка на кампанию"}, inplace=True)

    df_chart = df_chart.loc[:, [
        u"Кампания №", u"Название кампании", u"Дневной бюджет",
        u"Показы (спец.)", u"Клики (спец.)", u"Расходы (спец.)", u"CPC (спец.)", u"CTR (спец.)",
        u"Показы (юж./прав. блок)", u"Клики (юж./прав. блок)", u"Расходы (юж./прав. блок)",
        u"CPC (юж./прав. блок)",
        u"CTR (юж./прав. блок)",
        u"Показы (сети)", u"Клики (сети)", u"Расходы (сети)", u"CPC (сети)", u"CTR (сети)",
        u"Стратегия на поиске", u"Стратегия в сетях",
        u"Ссылка на кампанию"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Дневной бюджет": "float",
                                  u"Показы (спец.)": "int",
                                  u"Клики (спец.)": "int",
                                  u"Расходы (спец.)": "float",
                                  u"CPC (спец.)": "float",
                                  u"CTR (спец.)": "percent",
                                  u"Показы (юж./прав. блок)": "int",
                                  u"Клики (юж./прав. блок)": "int",
                                  u"Расходы (юж./прав. блок)": "float",
                                  u"CPC (юж./прав. блок)": "float",
                                  u"CTR (юж./прав. блок)": "percent",
                                  u"Показы (сети)": "int",
                                  u"Клики (сети)": "int",
                                  u"Расходы (сети)": "float",
                                  u"CPC (сети)": "float",
                                  u"CTR (сети)": "percent",
                                  u"Стратегия на поиске": "string",
                                  u"Стратегия в сетях": "string",
                                  u"Ссылка на кампанию": "string"})

    return df_chart


def table_used_strategies_report(df):
    """00_used_strategies"""

    df_chart = (df[["ShowsSpec", "ClicksSpec", "CostSpec",
                    "ShowsOther", "ClicksOther", "CostOther",
                    "ShowsRSYA", "ClicksRSYA", "CostRSYA",
                    "CampaignID", "Strategy"]]
                .assign(SearchStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Search"), "Search")))
                .assign(RSYAStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Network"), "Network")))
                .groupby(["SearchStrategy", "RSYAStrategy"], as_index=False)
                .agg({
                    "ShowsSpec": pd.np.sum,
                    "ClicksSpec": pd.np.sum,
                    "CostSpec": pd.np.sum,
                    "ShowsOther": pd.np.sum,
                    "ClicksOther": pd.np.sum,
                    "CostOther": pd.np.sum,
                    "ShowsRSYA": pd.np.sum,
                    "ClicksRSYA": pd.np.sum,
                    "CostRSYA": pd.np.sum,
                    "CampaignID": lambda col: col.unique().size
                })
                .sort_values("CampaignID", ascending=False))

    add_cpc_spec(df_chart)
    add_cpc_other(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_spec(df_chart)
    add_ctr_other(df_chart)
    add_ctr_rsya(df_chart)

    df_chart.rename(columns={
        "SearchStrategy": u"Стратегия на поиске",
        "RSYAStrategy": u"Стратегия в сетях",
        "ShowsSpec": u"Показы (спец.)",
        "ClicksSpec": u"Клики (спец.)",
        "CostSpec": u"Расходы (спец.)",
        "CPCSpec": u"CPC (спец.)",
        "CTRSpec": u"CTR (спец.)",
        "ShowsOther": u"Показы (юж./прав. блок)",
        "ClicksOther": u"Клики (юж./прав. блок)",
        "CostOther": u"Расходы (юж./прав. блок)",
        "CPCOther": u"CPC (юж./прав. блок)",
        "CTROther": u"CTR (юж./прав. блок)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Расходы (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)",
        "CampaignID": u"Кол-во кампаний"},
        inplace=True)

    df_chart = df_chart.loc[:, [
        u"Стратегия на поиске", u"Стратегия в сетях",
        u"Показы (спец.)", u"Клики (спец.)", u"Расходы (спец.)", u"CPC (спец.)", u"CTR (спец.)",
        u"Показы (юж./прав. блок)", u"Клики (юж./прав. блок)", u"Расходы (юж./прав. блок)",
        u"CPC (юж./прав. блок)", u"CTR (юж./прав. блок)",
        u"Показы (сети)", u"Клики (сети)", u"Расходы (сети)", u"CPC (сети)", u"CTR (сети)",
        u"Кол-во кампаний"
    ]]

    setattr(df_chart, "formats", {u"Стратегия на поиске": "string",
                                  u"Стратегия в сетях": "string",
                                  u"Показы (спец.)": "int",
                                  u"Клики (спец.)": "int",
                                  u"Расходы (спец.)": "float",
                                  u"CPC (спец.)": "float",
                                  u"CTR (спец.)": "percent",
                                  u"Показы (юж./прав. блок)": "int",
                                  u"Клики (юж./прав. блок)": "int",
                                  u"Расходы (юж./прав. блок)": "float",
                                  u"CPC (юж./прав. блок)": "float",
                                  u"CTR (юж./прав. блок)": "percent",
                                  u"Показы (сети)": "int",
                                  u"Клики (сети)": "int",
                                  u"Расходы (сети)": "float",
                                  u"CPC (сети)": "float",
                                  u"CTR (сети)": "percent",
                                  u"Кол-во кампаний": "int"})

    return df_chart


def table_campaigns_type(df, normalize):
    """04_campaigns_type"""

    df_chart = (df[["ShowsSearch", "ShowsRSYA", "Shows", "Clicks", "Cost", "CampaignID"]]
                .groupby("CampaignID", as_index=False)
                .agg({"ShowsSearch": pd.np.sum,
                      "ShowsRSYA": pd.np.sum,
                      "Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum}))
    df_chart['StrategyType'] = df_chart.apply(get_strategy_type, axis=1)
    df_chart = (df_chart[["Shows", "Clicks", "Cost", "CampaignID", "StrategyType"]]
                .groupby("StrategyType")
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "CampaignID": lambda x: x.unique().size}))
    if df_chart.shape[0] == 0:
        return None
    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart.rename(columns={
        "CampaignID": u"Кол-во кампаний",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": "CPC",
        "CTR": "CTR"},
        inplace=True)
    df_chart.rename(index={
        "only_search": u"Только поиск",
        "only_rsya": u"Только сети",
        "both": u"Все площадки"},
        inplace=True)
    df_chart.index.name = u"Стратегия показа"

    if normalize:
        sum_cols = [u"Кол-во кампаний", u"Показы", u"Клики", u"Расходы"]
        max_cols = [u"CPC", u"CTR"]
        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

        # Лимиты из головы эксперта, можно двигать
        try:
            rsya_cost_share = 1. - df_chart.loc[u"Только поиск", u"Расходы"]
        except KeyError:
            rsya_cost_share = 1.0
        if rsya_cost_share == 0.0:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="04_campaigns_type",
                title=u"Использование площадок",
                snippet=u"Сети не используются",
                importance_lvl=2)
            append_recommendation(df, Rec, uniq=False)

        elif rsya_cost_share < 0.05:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="04_campaigns_type",
                title=u"Использование площадок",
                snippet=u"Сети используются в незначительном объёме: %.0f%% от затрат" % (rsya_cost_share * 100.),
                importance_lvl=2)
            append_recommendation(df, Rec, uniq=False)

    df_chart = df_chart.loc[[u"Все площадки", u"Только поиск", u"Только сети"],
                            [u"Кол-во кампаний", u"Показы", u"Клики", u"Расходы", u"CPC", u"CTR"]].fillna('')

    if normalize:
        formats = {
            u"Кол-во кампаний": "percent",
            u"Показы": "percent",
            u"Клики": "percent",
            u"Расходы": "percent",
            u"CPC": "float",
            u"CTR": "float"}
    else:
        formats = {
            u"Кол-во кампаний": "int",
            u"Показы": "int",
            u"Клики": "int",
            u"Расходы": "float",
            u"CPC": "float",
            u"CTR": "percent"}

    setattr(df_chart, "formats", formats)

    return df_chart


def table_list_campaigns_type(df):
    """04_list_campaigns_type"""

    df_chart = (df[["CampaignID", "Cost",
                    "ShowsSearch", "ClicksSearch", "CostSearch",
                    "ShowsRSYA", "ClicksRSYA", "CostRSYA"]]
                .groupby("CampaignID", as_index=False)
                .agg({"Cost": pd.np.sum,
                      "ShowsSearch": pd.np.sum,
                      "ClicksSearch": pd.np.sum,
                      "CostSearch": pd.np.sum,
                      "ShowsRSYA": pd.np.sum,
                      "ClicksRSYA": pd.np.sum,
                      "CostRSYA": pd.np.sum})
                .assign(StrategyType=lambda x: x.apply(get_strategy_type, axis=1))
                .sort_values("Cost", ascending=False)
                .reset_index(drop=True)
                [["CampaignID", "StrategyType",
                  "ShowsSearch", "ClicksSearch", "CostSearch",
                  "ShowsRSYA", "ClicksRSYA", "CostRSYA"]])
    add_cpc_search(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_search(df_chart)
    add_ctr_rsya(df_chart)
    df_chart.StrategyType = df_chart.StrategyType.replace(to_replace={
        "only_search": u"Только поиск",
        "only_rsya": u"Только сети",
        "both": u"Все площадки"})
    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "StrategyType": u"Стратегия показа",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Расходы (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Расходы (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)"},
        inplace=True)

    df_chart = df_chart.loc[:, [u"Кампания №",
                                u"Стратегия показа",
                                u"Показы (поиск)", u"Клики (поиск)", u"Расходы (поиск)", u"CPC (поиск)", u"CTR (поиск)",
                                u"Показы (сети)", u"Клики (сети)", u"Расходы (сети)",
                                u"CPC (сети)", u"CTR (сети)"]].fillna("")

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Стратегия показа": "string",
                                  u"Показы (поиск)": "int",
                                  u"Клики (поиск)": "int",
                                  u"Расходы (поиск)": "float",
                                  u"CPC (поиск)": "float",
                                  u"CTR (поиск)": "percent",
                                  u"Показы (сети)": "int",
                                  u"Клики (сети)": "int",
                                  u"Расходы (сети)": "float",
                                  u"CPC (сети)": "float",
                                  u"CTR (сети)": "percent"})

    return df_chart
