# coding: utf-8
"""xlsx-keywords:
  - 06_templates_status
  - 06_templates_incorrect_usage
  - 06_display_url
  - 06_wo_display_url

Help:
[HasTemplate]
  (https://yandex.ru/support/direct/features/ad-templates.xml?lang=en)
"""
from ..review import Recommendation, append_recommendation
from ..const import EXCEL_MAX_ROWS
from ..utils import filter_by_adtype
from .utils import (
    add_cpc_search,
    add_cpc_rsya,
    add_cpc_desktop,
    add_ctr_search,
    add_ctr_rsya,
    add_ctr_desktop
)
import pandas as pd


def table_templates_status(df, normalize):
    """06_templates_status"""

    df_chart = filter_by_adtype(df, "TEXT_AD")
    df_chart = (df_chart[(df_chart.ShowsSearch > 0)]
                [["ShowsSearch", "ClicksSearch", "CostSearch",
                  "IsCorrectTemplate", "DirectBannerID", "DirectPhraseID"]]
                .assign(IsCorrectTemplate=lambda x: x.IsCorrectTemplate.astype(str))
                .groupby("IsCorrectTemplate", as_index=True)
                .agg({"ShowsSearch": pd.np.sum,
                      "ClicksSearch": pd.np.sum,
                      "CostSearch": pd.np.sum,
                      "DirectBannerID": lambda x: x.unique().size,
                      "DirectPhraseID": lambda x: x.unique().size})
                .fillna(0)
                .round(2)
                )

    if df_chart.shape[0] == 0:
        return None

    add_cpc_search(df_chart)
    add_ctr_search(df_chart)
    df_chart.rename(columns={
        "DirectBannerID": u"Кол-во объявлений",
        "DirectPhraseID": u"Кол-во фраз",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Расходы (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)"},
        inplace=True)
    df_chart.rename(index={
        "False": u"Используется фраза по умолчанию",
        "True": u"Фраза подставляется",
        "None": u"Без шаблона"},
        inplace=True)
    df_chart.index.names = [u"Корректность"]
    sum_cols = [u"Кол-во объявлений", u"Кол-во фраз", u"Показы (поиск)", u"Клики (поиск)", u"Расходы (поиск)"]
    max_cols = [u"CPC (поиск)", u"CTR (поиск)"]

    if normalize:
        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

        # Все пороги из головы эксперта, можно менять
        try:
            share_of_none = df_chart.loc[u"Без шаблона", u"Кол-во объявлений"]
        except KeyError:
            share_of_none = 0.
        if share_of_none > 0.05:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="06_templates_status",
                title=u"Использование шаблонов",
                snippet=u"В текстах объявлений не используются шаблоны: не менее %.0f%% объявлений" % (share_of_none * 100),
                importance_lvl=2)

            append_recommendation(df, Rec)
        try:
            share_of_false = df_chart.loc[u"Используется фраза по умолчанию", u"Кол-во фраз"]
        except KeyError:
            share_of_false = 0.
        if share_of_false > 0.05:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="06_templates_status",
                title=u"Использование шаблонов",
                snippet=u"Шаблоны используются, но фраза не подставляется: не менее %.0f%% фраз" % (share_of_false * 100),
                importance_lvl=2)

            append_recommendation(df, Rec)

    df_chart = df_chart.loc[[u"Без шаблона", u"Используется фраза по умолчанию", u"Фраза подставляется"],
                            sum_cols + max_cols].fillna('')

    if normalize:
        formats = {
            u"Кол-во объявлений": "percent",
            u"Кол-во фраз": "percent",
            u"Показы (поиск)": "percent",
            u"Клики (поиск)": "percent",
            u"Расходы (поиск)": "percent",
            u"CPC (поиск)": "float",
            u"CTR (поиск)": "float"}
    else:
        formats = {
            u"Кол-во объявлений": "int",
            u"Кол-во фраз": "int",
            u"Показы (поиск)": "int",
            u"Клики (поиск)": "int",
            u"Расходы (поиск)": "float",
            u"CPC (поиск)": "float",
            u"CTR (поиск)": "percent"}

    setattr(df_chart, "formats", formats)

    return df_chart


def table_templates_incorrect_usage(df):
    """06_templates_incorrect_usage"""

    df_chart = filter_by_adtype(df, "TEXT_AD")
    df_chart = (df_chart[
                  (df_chart.IsCorrectTemplate == False) &
                  (df_chart.ShowsSearch > 0)]  # noqa: E712
                .sort_values("ShowsSearch", ascending=False)
                .head(EXCEL_MAX_ROWS)
                .set_index(["DirectBannerID"])
                [["CampaignID", "CampaignName", "Title", "Body", "PhraseText", "ShowsSearch", "ClicksSearch", "CostSearch"]])
    try:
        df_chart.rename(index={
            "DirectBannerID": u"Баннер №"},
            inplace=True)
        df_chart.index.names = [u"Баннер №"]
    except:
        pass

    add_cpc_search(df_chart)
    add_ctr_search(df_chart)

    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "PhraseText": u"Ключевая фраза",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Расходы (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)"},
        inplace=True)

    df_chart = df_chart.loc[:, [u"Кампания №",
                                u"Название кампании",
                                u"Заголовок",
                                u"Объявление",
                                u"Ключевая фраза",
                                u"Показы (поиск)",
                                u"Клики (поиск)",
                                u"Расходы (поиск)",
                                u"CPC (поиск)",
                                u"CTR (поиск)"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Заголовок": "string",
                                  u"Объявление": "string",
                                  u"Ключевая фраза": "string",
                                  u"Показы (поиск)": "int",
                                  u"Клики (поиск)": "int",
                                  u"Расходы (поиск)": "float",
                                  u"CPC (поиск)": "float",
                                  u"CTR (поиск)": "percent"})

    return df_chart


def table_display_url_has_template(df, net_type, normalize):
    """06_display_url_х[R|S]

    Note:
      net_type : Search|RSYA
    """
    NET_VARIANTS = ("Search", "RSYA")
    assert net_type in NET_VARIANTS,\
           "`net_type` is not in %s" % NET_VARIANTS  # noqa: E127 (over-indented)
    df_chart = filter_by_adtype(df, "TEXT_AD")
    df_chart = df_chart[(df_chart["Shows%s" % net_type] > 0)]
    if net_type == "Search":
        net_text = u"на поиске"
    else:
        net_text = u"в сетях"
    if net_type == "Search":
        net_type = "Desktop"

    df_chart = (df_chart
                [["Shows%s" % net_type,
                  "Clicks%s" % net_type,
                  "Cost%s" % net_type,
                  "DisplayUrlHasTemplate",
                  "DirectBannerID",
                  "DirectPhraseID"]]
                .assign(DisplayUrlHasTemplate=lambda x: x.DisplayUrlHasTemplate.astype(str))
                .groupby("DisplayUrlHasTemplate", as_index=True)
                .agg({"Shows%s" % net_type: pd.np.sum,
                      "Clicks%s" % net_type: pd.np.sum,
                      "Cost%s" % net_type: pd.np.sum,
                      "DirectBannerID": lambda x: x.unique().size,
                      "DirectPhraseID": lambda x: x.unique().size})
                .fillna(0)
                .round(2)
                )
    if df_chart.shape[0] == 0:
        return None

    if net_type == "Desktop":
        add_cpc_desktop(df_chart)
        add_ctr_desktop(df_chart)
    if net_type == "RSYA":
        add_cpc_rsya(df_chart)
        add_ctr_rsya(df_chart)

    net_type_ru = {"Desktop": u"поиск", "RSYA": u"сети"}
    df_chart.rename(columns={
        "DirectBannerID": u"Кол-во объявлений",
        "DirectPhraseID": u"Кол-во фраз",
        "Shows%s" % net_type: u"Показы (%s)" % net_type_ru[net_type],
        "Clicks%s" % net_type: u"Клики (%s)" % net_type_ru[net_type],
        "Cost%s" % net_type: u"Расходы (%s)" % net_type_ru[net_type],
        "CPC%s" % net_type: u"CPC (%s)" % net_type_ru[net_type],
        "CTR%s" % net_type: u"CTR (%s)" % net_type_ru[net_type]},
        inplace=True)
    df_chart.rename(index={
        "False": u"Без шаблона",
        "True": u"С шаблоном",
        "None": u"Без отображаемой ссылки"},
        inplace=True)
    df_chart.index.names = [u"Использование отображаемой ссылки"]

    if normalize:
        sum_col = [u"Кол-во объявлений",
                   u"Кол-во фраз",
                   u"Показы (%s)" % net_type_ru[net_type],
                   u"Клики (%s)" % net_type_ru[net_type],
                   u"Расходы (%s)" % net_type_ru[net_type]]
        max_col = [u"CPC (%s)" % net_type_ru[net_type],
                   u"CTR (%s)" % net_type_ru[net_type]]
        df_chart.loc[:, sum_col] = df_chart.loc[:, sum_col].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_col] = df_chart.loc[:, max_col].apply(lambda x: x / x.max())

        try:
            share_of_none_b, share_of_none_s = df_chart.loc[
                u"Без отображаемой ссылки",
                [u"Кол-во объявлений", u"Показы (%s)" % net_type_ru[net_type]]]
        except KeyError:
            share_of_none_b, share_of_none_s = 0., 0.
        if share_of_none_b > 0.1 or share_of_none_s > 0.1:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="06_display_url_S" if net_type == "Desktop" else "06_display_url_R",
                title=u"Отображаемая ссылка",
                snippet=u"Отображаемая ссылка не настроена в %.0f%% объявлений для %.0f%% показов %s" %
                        (share_of_none_b * 100., share_of_none_s * 100., net_text),
                importance_lvl=2)

            append_recommendation(df, Rec)

    df_chart = df_chart.loc[[u"Без шаблона",
                             u"С шаблоном",
                             u"Без отображаемой ссылки"],
                            [u"Кол-во объявлений",
                             u"Кол-во фраз",
                             u"Показы (%s)" % net_type_ru[net_type],
                             u"Клики (%s)" % net_type_ru[net_type],
                             u"Расходы (%s)" % net_type_ru[net_type],
                             u"CPC (%s)" % net_type_ru[net_type],
                             u"CTR (%s)" % net_type_ru[net_type]]].fillna('')

    if normalize:
        formats = {
            u"Кол-во объявлений": "percent",
            u"Кол-во фраз": "percent",
            u"Показы (%s)" % net_type_ru[net_type]: "percent",
            u"Клики (%s)" % net_type_ru[net_type]: "percent",
            u"Расходы (%s)" % net_type_ru[net_type]: "percent",
            u"CPC (%s)" % net_type_ru[net_type]: "float",
            u"CTR (%s)" % net_type_ru[net_type]: "float"}
    else:
        formats = {
            u"Кол-во объявлений": "int",
            u"Кол-во фраз": "int",
            u"Показы (%s)" % net_type_ru[net_type]: "int",
            u"Клики (%s)" % net_type_ru[net_type]: "int",
            u"Расходы (%s)" % net_type_ru[net_type]: "float",
            u"CPC (%s)" % net_type_ru[net_type]: "float",
            u"CTR (%s)" % net_type_ru[net_type]: "percent"}

    setattr(df_chart, "formats", formats)

    return df_chart


def table_wo_display_url(df, sort_criteria):
    """06_wo_display_url

    Note:
      sort_criteria : ShowsSearch|ShowsRSYA
    """
    CRITERIA_VARIANTS = ("ShowsSearch", "ShowsRSYA")
    assert sort_criteria in CRITERIA_VARIANTS,\
           "`sort_criteria` is not in %s" % CRITERIA_VARIANTS  # noqa: E127 (over-indented)
    df_chart = filter_by_adtype(df, "TEXT_AD")

    if sort_criteria == "ShowsSearch":
        df_chart = (df_chart[
                      df_chart.DisplayUrlHasTemplate.isnull() &
                      (df_chart[sort_criteria] > 0)]  # noqa: E712
                    .sort_values("ShowsDesktop", ascending=False)
                    .head(EXCEL_MAX_ROWS)
                    .set_index(["DirectBannerID"])
                    [["CampaignID", "CampaignName", "Title", "Body", "PhraseText",
                      "ShowsDesktop", "ClicksDesktop", "CostDesktop",
                      "ShowsRSYA", "ClicksRSYA", "CostRSYA"]])
        df_chart.rename(
            columns={
                "ShowsDesktop": "ShowsSearch",
                "ClicksDesktop": "ClicksSearch",
                "CostDesktop": "CostSearch"},
            inplace=True)

    else:
        df_chart = (df_chart[
                      df_chart.DisplayUrlHasTemplate.isnull() &
                      (df_chart[sort_criteria] > 0)]  # noqa: E712
                    .sort_values(sort_criteria, ascending=False)
                    .head(EXCEL_MAX_ROWS)
                    .set_index(["DirectBannerID"])
                    [["CampaignID", "CampaignName", "Title", "Body", "PhraseText",
                      "ShowsSearch", "ClicksSearch", "CostSearch",
                      "ShowsRSYA", "ClicksRSYA", "CostRSYA"]])
    if df_chart.shape[0] == 0:
        return None

    add_cpc_search(df_chart)
    add_ctr_search(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_rsya(df_chart)

    try:
        df_chart.rename(index={
            "DirectBannerID": u"Баннер №"},
            inplace=True)
        df_chart.index.names = [u"Баннер №"]
    except:
        pass

    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "PhraseText": u"Ключевая фраза",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Расходы (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Расходы (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)"},
        inplace=True)

    df_chart = df_chart.loc[:, [u"Кампания №",
                                u"Название кампании",
                                u"Заголовок",
                                u"Объявление",
                                u"Ключевая фраза",
                                u"Показы (поиск)",
                                u"Клики (поиск)",
                                u"Расходы (поиск)",
                                u"CPC (поиск)",
                                u"CTR (поиск)",
                                u"Показы (сети)",
                                u"Клики (сети)",
                                u"Расходы (сети)",
                                u"CPC (сети)",
                                u"CTR (сети)"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Заголовок": "string",
                                  u"Объявление": "string",
                                  u"Ключевая фраза": "string",
                                  u"Показы (поиск)": "int",
                                  u"Клики (поиск)": "int",
                                  u"Расходы (поиск)": "float",
                                  u"CPC (поиск)": "float",
                                  u"CTR (поиск)": "percent",
                                  u"Показы (сети)": "int",
                                  u"Клики (сети)": "int",
                                  u"Расходы (сети)": "float",
                                  u"CPC (сети)": "float",
                                  u"CTR (сети)": "percent"})

    return df_chart
