# coding: utf-8
"""xlsx-keywords:
  - 14_vcard_usage
  - 14_banners_wo_vcard

Help:
[vCard](https://yandex.com/support/direct/features/vcards.xml)
"""
from ..review import Recommendation, append_recommendation
from ..const import EXCEL_MAX_ROWS
from ..utils import filter_by_adtype
from .utils import (
    add_cpc_total,
    add_ctr_total
)
import pandas as pd


def table_vcard_usage(df, normalize):
    """14_vcard_usage"""

    df_chart = filter_by_adtype(df, "TEXT_AD")
    df_chart = (df_chart[(df_chart.Shows > 0)]
                [["Shows", "Clicks", "Cost",
                  "VisitCardUsage", "DirectBannerID", "DirectPhraseID"]]
                .assign(VisitCardUsage=lambda x: x.VisitCardUsage.astype(str))
                .groupby("VisitCardUsage", as_index=True)
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "DirectBannerID": lambda x: x.unique().size,
                      "DirectPhraseID": lambda x: x.unique().size})
                .fillna(0)
                .round(2))

    if df_chart.shape[0] == 0:
        return None

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart.rename(columns={
        "DirectBannerID": u"Кол-во объявлений",
        "DirectPhraseID": u"Кол-во фраз",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": u"CPC",
        "CTR": u"CTR"},
        inplace=True)
    df_chart.rename(index={
        "False": u"Без доп. полей",
        "True": u"С доп. полями",
        "None": u"Не используется"},
        inplace=True)
    df_chart.index.names = [u"Использование визитки"]

    if normalize:
        sum_cols = [u"Кол-во объявлений", u"Кол-во фраз", u"Показы", u"Клики", u"Расходы"]
        max_cols = [u"CPC", u"CTR"]

        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

        # Все лимиты из головы эксперта, можно менять
        try:
            vcard_none_b, vcard_none_s = df_chart.loc[u"Не используется", [u"Кол-во объявлений", u"Показы"]]
        except KeyError:
            vcard_none_b, vcard_none_s = 0., 0.
        if vcard_none_b > 0.1 or vcard_none_s > 0.1:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="14_vcard_usage",
                title=u"Виртуальные визитки",
                snippet=u"Нет виртуальных визиток: %.0f%% объявлений (%.0f%% показов)" % (vcard_none_b * 100., vcard_none_s * 100.),
                importance_lvl=2)

            append_recommendation(df, Rec, uniq=False)

        # Все лимиты из головы эксперта, можно менять
        try:
            vcard_semiempty_b, vcard_semiempty_s = df_chart.loc[u"Без доп. полей", [u"Кол-во объявлений", u"Показы"]]
        except KeyError:
            vcard_semiempty_b, vcard_semiempty_s = 0., 0.
        if vcard_semiempty_b > 0.1 or vcard_semiempty_s > 0.1:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="14_vcard_usage",
                title=u"Виртуальные визитки",
                snippet=u"Не заполнены дополнительные поля (адрес, описание, e-mail, ОГРН) у визиток: %.0f%% объявлений (%.0f%% показов)" %
                        (vcard_semiempty_b * 100., vcard_semiempty_s * 100.),
                importance_lvl=2)

            append_recommendation(df, Rec, uniq=False)

    df_chart = df_chart.loc[[u"Не используется",
                             u"Без доп. полей",
                             u"С доп. полями"],
                            [u"Кол-во объявлений",
                             u"Кол-во фраз",
                             u"Показы",
                             u"Клики",
                             u"Расходы",
                             u"CPC",
                             u"CTR"]].fillna('')

    if normalize:
        formats = {
            u"Кол-во объявлений": "percent",
            u"Кол-во фраз": "percent",
            u"Показы": "percent",
            u"Клики": "percent",
            u"Расходы": "percent",
            u"CPC": "float",
            u"CTR": "float"}
    else:
        formats = {
            u"Кол-во объявлений": "int",
            u"Кол-во фраз": "int",
            u"Показы": "int",
            u"Клики": "int",
            u"Расходы": "float",
            u"CPC": "float",
            u"CTR": "percent"}

    setattr(df_chart, "formats", formats)

    return df_chart


def table_banners_wo_vcard(df):
    """14_banners_wo_vcard"""

    df_chart = filter_by_adtype(df, "TEXT_AD")
    df_chart = (df_chart[
                  df_chart.VisitCardUsage.isnull() &
                  (df_chart.Shows > 0)]  # noqa: E712
                .sort_values("Shows", ascending=False)
                .head(EXCEL_MAX_ROWS)
                .set_index(["DirectBannerID"])
                .loc[:, ["CampaignID", "CampaignName", "Title", "Body", "PhraseText",
                         "Shows", "Clicks", "Cost"]])

    if df_chart.shape[0] == 0:
        return None

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    try:
        df_chart.rename(index={
            "DirectBannerID": u"Баннер №"},
            inplace=True)
        df_chart.index.names = [u"Баннер №"]
    except:
        pass

    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "PhraseText": u"Ключевая фраза",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": u"CPC",
        "CTR": u"CTR"},
        inplace=True)

    df_chart = df_chart.loc[:, [u"Кампания №",
                                u"Название кампании",
                                u"Заголовок",
                                u"Объявление",
                                u"Ключевая фраза",
                                u"Показы",
                                u"Клики",
                                u"Расходы",
                                u"CPC",
                                u"CTR"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Заголовок": "string",
                                  u"Объявление": "string",
                                  u"Ключевая фраза": "string",
                                  u"Показы": "int",
                                  u"Клики": "int",
                                  u"Расходы": "float",
                                  u"CPC": "float",
                                  u"CTR": "percent"})

    return df_chart
