# coding: utf-8
"""xlsx-mobile:
  - 00_table_total_campaigns_stat
  - 00_used_strategies
  - 04_campaigns_type
  - 04_list_campaigns_type

Help:
[Раздельное управление площадками]
  (https://yandex.ru/support/direct/strategies/independent.xml?lang=en)
[Creating a separate campaign for displaying ads in the ad networks]
  (https://yandex.ru/support/direct/efficiency/content-websites.xml?lang=en)
"""
from ..xlsx_keywords.campaign_stat import extract_strategy, translate_strategy_type
from ..xlsx_chart.chart import Chart, chart_add_series, FONT_NAME
from ..utils import get_cid, get_strategy_type
from .utils import (
    add_cpc_search,
    add_cpc_spec,
    add_cpc_other,
    add_cpc_rsya,
    add_cpc_total,
    add_ctr_search,
    add_ctr_spec,
    add_ctr_other,
    add_ctr_rsya,
    add_ctr_total
)
import pandas as pd


def table_total_campaigns_stat(df):
    """00_total_campaigns_stat"""

    df_chart = (df[["ShowsSpec", "ClicksSpec", "CostSpec",
                    "ShowsOther", "ClicksOther", "CostOther",
                    "ShowsRSYA", "ClicksRSYA", "CostRSYA",
                    "CampaignID", "CampaignName", "DayBudget", "Strategy"]]
                .groupby(["CampaignID", "CampaignName", "DayBudget", "Strategy"], as_index=False)
                .sum()
                .assign(SearchStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Search"), "Search")))
                .assign(RSYAStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Network"), "Network")))
                .assign(HelperUrl=lambda df: df.CampaignID.apply(get_cid))
                .drop("Strategy", axis=1))
    add_cpc_spec(df_chart)
    add_cpc_other(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_spec(df_chart)
    add_ctr_other(df_chart)
    add_ctr_rsya(df_chart)
    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "DayBudget": u"Дневной бюджет",
        "ShowsSpec": u"Показы (спец.)",
        "ClicksSpec": u"Клики (спец.)",
        "CostSpec": u"Затраты (спец.)",
        "CPCSpec": u"CPC (спец.)",
        "CTRSpec": u"CTR (спец.)",
        "ShowsOther": u"Показы (юж./прав. блок)",
        "ClicksOther": u"Клики (юж./прав. блок)",
        "CostOther": u"Затраты (юж./прав. блок)",
        "CPCOther": u"CPC (юж./прав. блок)",
        "CTROther": u"CTR (юж./прав. блок)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Затраты (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)",
        "SearchStrategy": u"Стратегия на поиске",
        "RSYAStrategy": u"Стратегия в сетях",
        "HelperUrl": u"Ссылка на кампанию"}, inplace=True)

    return df_chart.loc[:, [
        u"Кампания №",
        u"Название кампании",
        u"Дневной бюджет",
        u"Показы (спец.)",
        u"Клики (спец.)",
        u"Затраты (спец.)",
        u"CPC (спец.)",
        u"CTR (спец.)",
        u"Показы (юж./прав. блок)",
        u"Клики (юж./прав. блок)",
        u"Затраты (юж./прав. блок)",
        u"CPC (юж./прав. блок)",
        u"CTR (юж./прав. блок)",
        u"Показы (сети)",
        u"Клики (сети)",
        u"Затраты (сети)",
        u"CPC (сети)",
        u"CTR (сети)",
        u"Стратегия на поиске",
        u"Стратегия в сетях",
        u"Ссылка на кампанию"]]


def table_used_strategies_report(df):
    """00_used_strategies"""

    df_chart = (df[["ShowsSpec", "ClicksSpec", "CostSpec",
                    "ShowsOther", "ClicksOther", "CostOther",
                    "ShowsRSYA", "ClicksRSYA", "CostRSYA",
                    "CampaignID", "Strategy"]]
                .assign(SearchStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Search"), "Search")))
                .assign(RSYAStrategy=lambda df: df.Strategy.apply(
                    lambda x: translate_strategy_type(extract_strategy(x, "Network"), "Network")))
                .groupby(["SearchStrategy", "RSYAStrategy"], as_index=False)
                .agg({
                    "ShowsSpec": pd.np.sum,
                    "ClicksSpec": pd.np.sum,
                    "CostSpec": pd.np.sum,
                    "ShowsOther": pd.np.sum,
                    "ClicksOther": pd.np.sum,
                    "CostOther": pd.np.sum,
                    "ShowsRSYA": pd.np.sum,
                    "ClicksRSYA": pd.np.sum,
                    "CostRSYA": pd.np.sum,
                    "CampaignID": lambda col: col.unique().size
                })
                .sort_values("CampaignID", ascending=False))
    add_cpc_spec(df_chart)
    add_cpc_other(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_spec(df_chart)
    add_ctr_other(df_chart)
    add_ctr_rsya(df_chart)

    df_chart.rename(columns={
        "SearchStrategy": u"Стратегия на поиске",
        "RSYAStrategy": u"Стратегия в сетях",
        "ShowsSpec": u"Показы (спец.)",
        "ClicksSpec": u"Клики (спец.)",
        "CostSpec": u"Затраты (спец.)",
        "CPCSpec": u"CPC (спец.)",
        "CTRSpec": u"CTR (спец.)",
        "ShowsOther": u"Показы (юж./прав. блок)",
        "ClicksOther": u"Клики (юж./прав. блок)",
        "CostOther": u"Затраты (юж./прав. блок)",
        "CPCOther": u"CPC (юж./прав. блок)",
        "CTROther": u"CTR (юж./прав. блок)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Затраты (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)",
        "CampaignID": u"Кол-во кампаний"
    }, inplace=True)

    return df_chart.loc[:, [
        u"Стратегия на поиске",
        u"Стратегия в сетях",
        u"Показы (спец.)",
        u"Клики (спец.)",
        u"Затраты (спец.)",
        u"CPC (спец.)",
        u"CTR (спец.)",
        u"Показы (юж./прав. блок)",
        u"Клики (юж./прав. блок)",
        u"Затраты (юж./прав. блок)",
        u"CPC (юж./прав. блок)",
        u"CTR (юж./прав. блок)",
        u"Показы (сети)",
        u"Клики (сети)",
        u"Затраты (сети)",
        u"CPC (сети)",
        u"CTR (сети)",
        u"Кол-во кампаний"]]


def table_campaigns_type(df, normalize):
    """04_campaigns_type"""

    df_chart = (df[["ShowsSearch", "ShowsRSYA", "Shows", "Clicks", "Cost", "CampaignID"]]
                .groupby("CampaignID", as_index=False)
                .agg({"ShowsSearch": pd.np.sum,
                      "ShowsRSYA": pd.np.sum,
                      "Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum}))
    df_chart['StrategyType'] = df_chart.apply(get_strategy_type, axis=1)
    df_chart = (df_chart[["Shows", "Clicks", "Cost", "CampaignID", "StrategyType"]]
                .groupby("StrategyType")
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "CampaignID": lambda x: x.unique().size}))
    if df_chart.shape[0] == 0:
        return None
    add_cpc_total(df_chart)
    add_ctr_total(df_chart)
    df_chart.rename(columns={
        "CampaignID": u"Кол-во кампаний",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Затраты",
        "CPC": "CPC",
        "CTR": "CTR"},
        inplace=True)
    df_chart.rename(index={
        "only_search": u"Только поиск",
        "only_rsya": u"Только сети",
        "both": u"Все площадки"},
        inplace=True)
    df_chart.index.name = u"Стратегия показа"
    if normalize:
        sum_cols = [u"Кол-во кампаний", u"Показы", u"Клики", u"Затраты"]
        max_cols = [u"CPC", u"CTR"]
        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

    return df_chart.loc[[u"Все площадки",
                         u"Только поиск",
                         u"Только сети"],
                        [u"Кол-во кампаний",
                         u"Показы",
                         u"Клики",
                         u"Затраты",
                         u"CPC",
                         u"CTR"]
                        ].fillna('')


def plot_campaigns_type(writer, sheet_name, df):
    """(ExcelWriter, str, DataFrame) -> ExcelWriter"""
    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "bar", "percent_stacked", u'Разделение по площадкам')
    chart.set_y_axis({"name": u"Доля от общего",
                      'name_font': {"size": 16, "bold": False, "name": FONT_NAME},
                      'num_font': {"size": 14, "bold": False, "name": FONT_NAME},
                      'num_format': '#0%',
                      'major_tick_mark': 'none',
                      'minor_tick_mark': 'none',
                      'line': {'none': True}})
    positions = ("inside_base", "center")
    series = range(3)
    for x in series:
        irow = 2 + x  # 2 is offset to values
        pos_type = positions[x == series[0] or x == series[-1]]
        chart_add_series(chart, x, {
            'values': u"='{0}'!B{1}:E{1}".format(sheet_name, irow),
            "categories": u"='{}'!B1:E1".format(sheet_name),
            "name": u"='{0}'!A{1}".format(sheet_name, irow),
            "data_labels": {
                "position": pos_type,
                "num_format": "#0%"}})

    sheet.insert_chart('A8', chart)

    return writer


def plot_campaigns_type_ctr_cpc(writer, sheet_name, df):
    """(ExcelWriter, str, DataFrame) -> ExcelWriter"""

    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "column", None, u"Показатели работы")

    chart.set_y_axis({
        "visible": True,
        "name": u"",  # CPC
        'name_font': {
            "size": 16,
            "bold": False,
            "name": FONT_NAME
        },
        'num_font': {
            "size": 14,
            "bold": False,
            "name": FONT_NAME
        },
        'num_format': '#0.0',
        'major_tick_mark': 'none',
        'minor_tick_mark': 'none',
        'line': {
            'none': True
        },
        "label_position": "none",
        "min": 0,
    })
    chart.set_y2_axis({
        "visible": True,
        "name": u"",  # CTR
        'name_font': {
            "size": 16,
            "bold": False,
            "name": FONT_NAME
        },
        'num_font': {
            "size": 14,
            "bold": False,
            "name": FONT_NAME
        },
        'num_format': '#0.0%',
        'major_tick_mark': 'none',
        'minor_tick_mark': 'none',
        'line': {
            'none': True
        },
        "label_position": "none",
        "min": 0,
    })

    last_row = df.index.size + 1
    # CPC (поиск)
    chart_add_series(chart, 0, {
        'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, "O", last_row),
        "categories": u"='{}''!J2:J{}".format(sheet_name, last_row),
        "name": u"='{0}'!{1}1".format(sheet_name, "O"),
        "data_labels": {
            "num_format": "#0.0"
        },
        "overlap": 0,  # The range is +/- 100. The default is 0
        "gap": 340  # The range is 0 to 500. The default is 150
    })
    # fake series for a margin
    chart_add_series(chart, 1, {
        'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, "Q", last_row),
        "categories": u"='{}'!J2:J{}".format(sheet_name, last_row),
        "name": u"='{0}'!{1}1".format(sheet_name, "Q"),
        "data_labels": {
            "num_format": "#0.0%"
        },
        "y2_axis": 1
    })
    # CTR (поиск)
    chart_add_series(chart, 1, {
        'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, "P", last_row),
        "categories": u"='{}'!J2:J{}".format(sheet_name, last_row),
        "name": u"='{0}'!{1}1".format(sheet_name, "P"),
        "data_labels": {
            "num_format": "#0.0%"
        },
        "overlap": -100,
        "gap": 150,
        "y2_axis": 1
    })
    chart.set_legend({
        'position': 'bottom',
        "font": {'size': 14,
                 'bold': False,
                 'name': FONT_NAME},
        'delete_series': [1, 1]})
    sheet.insert_chart("P8", chart)

    return writer


def table_list_campaigns_type(df):
    """04_list_campaigns_type"""

    df_chart = (df[["CampaignID", "Cost",
                    "ShowsSearch", "ClicksSearch", "CostSearch",
                    "ShowsRSYA", "ClicksRSYA", "CostRSYA"]]
                .groupby("CampaignID", as_index=False)
                .agg({"Cost": pd.np.sum,
                      "ShowsSearch": pd.np.sum,
                      "ClicksSearch": pd.np.sum,
                      "CostSearch": pd.np.sum,
                      "ShowsRSYA": pd.np.sum,
                      "ClicksRSYA": pd.np.sum,
                      "CostRSYA": pd.np.sum})
                .assign(StrategyType=lambda x: x.apply(get_strategy_type, axis=1))
                .sort_values("Cost", ascending=False)
                .reset_index(drop=True)
                [["CampaignID", "StrategyType",
                  "ShowsSearch", "ClicksSearch", "CostSearch",
                  "ShowsRSYA", "ClicksRSYA", "CostRSYA"]])
    add_cpc_search(df_chart)
    add_cpc_rsya(df_chart)
    add_ctr_search(df_chart)
    add_ctr_rsya(df_chart)
    df_chart.StrategyType = df_chart.StrategyType.replace(to_replace={
        "only_search": u"Только поиск",
        "only_rsya": u"Только сети",
        "both": u"Все площадки"})
    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "StrategyType": u"Стратегия показа",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Затраты (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)",
        "ShowsRSYA": u"Показы (сети)",
        "ClicksRSYA": u"Клики (сети)",
        "CostRSYA": u"Затраты (сети)",
        "CPCRSYA": u"CPC (сети)",
        "CTRRSYA": u"CTR (сети)"},
        inplace=True)

    return df_chart.loc[:, [u"Кампания №",
                            u"Стратегия показа",
                            u"Показы (поиск)",
                            u"Клики (поиск)",
                            u"Затраты (поиск)",
                            u"CPC (поиск)",
                            u"CTR (поиск)",
                            u"Показы (сети)",
                            u"Клики (сети)",
                            u"Затраты (сети)",
                            u"CPC (сети)",
                            u"CTR (сети)"]].fillna(0)
