# coding: utf-8
"""xlsx-mobile:
  - 06_templates_status
  - 06_templates_incorrect_usage

Help:
В РМП нет отображаемой ссылки
[HasTemplate]
  (https://yandex.ru/support/direct/features/ad-templates.xml?lang=en)
"""
from ..const import EXCEL_MAX_ROWS
from ..xlsx_chart.chart import Chart, chart_add_series, FONT_NAME
from ..utils import filter_by_adtype
from .utils import (
    add_cpc_search,
    add_ctr_search
)
import pandas as pd


def table_templates_status(df, normalize):
    """06_templates_status"""

    df_chart = filter_by_adtype(df, "MOBILE_APP_AD")
    df_chart = (df_chart[(df_chart.ShowsSearch > 0)]
                [["ShowsSearch", "ClicksSearch", "CostSearch",
                  "IsCorrectTemplate", "DirectBannerID", "DirectPhraseID"]]
                .assign(IsCorrectTemplate=lambda x: x.IsCorrectTemplate.astype(str))
                .groupby("IsCorrectTemplate", as_index=True)
                .agg({"ShowsSearch": pd.np.sum,
                      "ClicksSearch": pd.np.sum,
                      "CostSearch": pd.np.sum,
                      "DirectBannerID": lambda x: x.unique().size,
                      "DirectPhraseID": lambda x: x.unique().size})
                .fillna(0)
                .round(2)
                )

    if df_chart.shape[0] == 0:
        return None

    add_cpc_search(df_chart)
    add_ctr_search(df_chart)
    df_chart.rename(columns={
        "DirectBannerID": u"Кол-во баннеров",
        "DirectPhraseID": u"Кол-во фраз",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Затраты (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)"},
        inplace=True)
    df_chart.rename(index={
        "False": u"Используется фраза по-умолчанию",
        "True": u"Фраза подставляется",
        "None": u"Без шаблона"},
        inplace=True)
    df_chart.index.names = [u"Корректность"]

    if normalize:
        sum_cols = [u"Кол-во баннеров", u"Кол-во фраз", u"Показы (поиск)", u"Клики (поиск)", u"Затраты (поиск)"]
        max_cols = [u"CPC (поиск)", u"CTR (поиск)"]

        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

    return df_chart.loc[[u"Без шаблона",
                         u"Используется фраза по-умолчанию",
                         u"Фраза подставляется"],
                        [u"Кол-во баннеров",
                         u"Кол-во фраз",
                         u"Показы (поиск)",
                         u"Клики (поиск)",
                         u"Затраты (поиск)",
                         u"CPC (поиск)",
                         u"CTR (поиск)"]].fillna('')


def plot_templates_status(writer, sheet_name, df):
    """(ExcelWriter, str, DataFrame) -> ExcelWriter"""
    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "bar", "percent_stacked", u'Использование шаблона')
    chart.set_y_axis({'num_font': {"size": 14, "bold": False, "name": FONT_NAME},
                      'num_format': '#0%',
                      'major_tick_mark': 'none',
                      'minor_tick_mark': 'none',
                      'line': {'none': True}})
    positions = ("inside_base", "center")
    series = range(3)
    for x in series:
        irow = 2 + x  # 2 is offset to values
        pos_type = positions[x == series[0] or x == series[-1]]
        chart_add_series(chart, x, {
            'values': u"='{0}'!B{1}:E{1}".format(sheet_name, irow),
            "categories": u"='{}'!B1:E1".format(sheet_name),
            "name": u"='{0}'!A{1}".format(sheet_name, irow),
            "data_labels": {
                "position": pos_type,
                "num_format": "#0%"}})

    sheet.insert_chart('A8', chart)

    return writer


def plot_templates_status_ctr_cpc(writer, sheet_name, df):
    """(ExcelWriter, str, DataFrame) -> ExcelWriter"""

    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "column", None, u"Показатели работы")

    chart.set_y_axis({
        "visible": True,
        "name": u"",  # CPC
        'name_font': {
            "size": 16,
            "bold": False,
            "name": FONT_NAME
        },
        'num_font': {
            "size": 14,
            "bold": False,
            "name": FONT_NAME
        },
        'num_format': '#0.0',
        'major_tick_mark': 'none',
        'minor_tick_mark': 'none',
        'line': {
            'none': True
        },
        "label_position": "none",
        "min": 0,
    })
    chart.set_y2_axis({
        "visible": True,
        "name": u"",  # CTR
        'name_font': {
            "size": 16,
            "bold": False,
            "name": FONT_NAME
        },
        'num_font': {
            "size": 14,
            "bold": False,
            "name": FONT_NAME
        },
        'num_format': '#0.0%',
        'major_tick_mark': 'none',
        'minor_tick_mark': 'none',
        'line': {
            'none': True
        },
        "label_position": "none",
        "min": 0,
    })

    last_row = df.index.size + 1
    # CPC (поиск)
    chart_add_series(chart, 0, {
        'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, "Q", last_row),
        "categories": u"='{}'!K2:K{}".format(sheet_name, last_row),
        "name": u"='{0}'!{1}1".format(sheet_name, "Q"),
        "data_labels": {
            "num_format": "#0.0"
        },
        "overlap": 0,  # The range is +/- 100. The default is 0
        "gap": 340  # The range is 0 to 500. The default is 150
    })
    # fake series for a margin
    chart_add_series(chart, 1, {
        'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, "S", last_row),
        "categories": u"='{}'!K2:K{}".format(sheet_name, last_row),
        "name": u"='{0}'!{1}1".format(sheet_name, "S"),
        "data_labels": {
            "num_format": "#0.0%"
        },
        "y2_axis": 1
    })
    # CTR (поиск)
    chart_add_series(chart, 1, {
        'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, "R", last_row),
        "categories": u"='{}'!K2:K{}".format(sheet_name, last_row),
        "name": u"='{0}'!{1}1".format(sheet_name, "R"),
        "data_labels": {
            "num_format": "#0.0%"
        },
        "overlap": -100,
        "gap": 150,
        "y2_axis": 1
    })
    chart.set_legend({
        'position': 'bottom',
        "font": {'size': 14,
                 'bold': False,
                 'name': FONT_NAME},
        'delete_series': [1, 1]})
    sheet.insert_chart("P8", chart)

    return writer


def table_templates_incorrect_usage(df):
    """06_templates_incorrect_usage"""

    df_chart = filter_by_adtype(df, "MOBILE_APP_AD")
    df_chart = (df_chart[
                  (df_chart.IsCorrectTemplate == False) &
                  (df_chart.ShowsSearch > 0)]  # noqa: E712
                .sort_values("ShowsSearch", ascending=False)
                .head(EXCEL_MAX_ROWS)
                .set_index(["DirectBannerID"])
                [["CampaignID", "CampaignName", "Title", "Body", "PhraseText", "ShowsSearch", "ClicksSearch", "CostSearch"]])

    try:
        df_chart.rename(index={
            "DirectBannerID": u"Баннер №"},
            inplace=True)
        df_chart.index.names = [u"Баннер №"]
    except:
        pass

    add_cpc_search(df_chart)
    add_ctr_search(df_chart)
    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "PhraseText": u"Ключевая фраза",
        "ShowsSearch": u"Показы (поиск)",
        "ClicksSearch": u"Клики (поиск)",
        "CostSearch": u"Затраты (поиск)",
        "CPCSearch": u"CPC (поиск)",
        "CTRSearch": u"CTR (поиск)"},
        inplace=True)

    return df_chart.loc[:, [u"Кампания №",
                            u"Название кампании",
                            u"Заголовок",
                            u"Объявление",
                            u"Ключевая фраза",
                            u"Показы (поиск)",
                            u"Клики (поиск)",
                            u"Затраты (поиск)",
                            u"CPC (поиск)",
                            u"CTR (поиск)"]]
