# coding: utf-8
"""xlsx-queries:
  - 01_top_non-bolded_queries
  - 01_highlight_hist_queries
  - 04_dssm_score_hist_queries
  - 04_top_irrel_queries_x_y
  - 04_top_irrel_campaigns
  - 04_eshows_share_queries
  - 04_top_eshows_share_queries
  - 04_top_eshows_share_campaigns
"""
from ..review import Recommendation, append_recommendation
from ..xlsx_keywords.utils import add_cpc_total, add_ctr_total
from ..xlsx_chart.chart import Chart, chart_add_series, FONT_NAME, FONT_SIZE
from ..utils import get_cid_bid, get_cid
from ..const import EXCEL_MAX_ROWS
import pandas as pd


def table_top_nonbolded_queries(df):
    """01_top_nonbolded_queries"""

    df_chart = df[[
        "CampaignID", "CampaignName", "DirectBannerID", "Title", "Body", "Query",
        "PhraseText", "QueryHighlighTitleBody", "QueryNonBoldedWordsTitleBody",
        "QueryWordsCnt", "Shows", "Clicks", "Cost"]].copy()
    df_chart = df_chart.loc[
        (df_chart.QueryWordsCnt != 0) &
        (df_chart.Shows > 0) &
        (df_chart.QueryNonBoldedWordsTitleBody != ""), :]

    if df_chart.shape[0] == 0:
        return None

    df_chart["QueryBoldShare"] = df_chart.QueryHighlighTitleBody / df_chart.QueryWordsCnt
    df_chart.drop(["QueryHighlighTitleBody", "QueryWordsCnt"], axis=1, inplace=True)
    df_chart['ShowsNonBolded'] = ((1 - df_chart['QueryBoldShare']) * df_chart['Shows']).astype(int)
    df_chart = df_chart.sort_values("ShowsNonBolded", ascending=False).head(EXCEL_MAX_ROWS)
    df_chart['HelperURL'] = df_chart.apply(
        lambda row: get_cid_bid(row["CampaignID"], row["DirectBannerID"]),
        axis=1)

    df_chart = df_chart[[
        "CampaignID", "CampaignName", "DirectBannerID", "Title", "Body", "Query", "PhraseText",
        "QueryNonBoldedWordsTitleBody", "Shows", "Clicks", "Cost", "HelperURL"]]

    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "DirectBannerID": u"Баннер №",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "Query": u"Поисковый запрос",
        "PhraseText": u"Условие показа",
        "QueryNonBoldedWordsTitleBody": u"Неподсвеченные слова",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "HelperURL": u"Ссылка на баннер"},
        inplace=True)

    df_chart = df_chart[[
        u"Кампания №", u"Название кампании", u"Баннер №", u"Заголовок", u"Объявление",
        u"Поисковый запрос", u"Условие показа", u"Неподсвеченные слова",
        u"Показы", u"Клики", u"Расходы", u"Ссылка на баннер"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Баннер №": "string",
                                  u"Заголовок": "string",
                                  u"Объявление": "string",
                                  u"Поисковый запрос": "string",
                                  u"Условие показа": "string",
                                  u"Неподсвеченные слова": "string",
                                  u"Показы": "int",
                                  u"Клики": "int",
                                  u"Расходы": "float",
                                  u"Ссылка на баннер": "string"})

    return df_chart


def table_highlight_queries(df, normalize):
    """01_highlight_hist_queries"""

    df_chart = df[["QueryHighlighTitleBody", "QueryWordsCnt", "Shows", "Clicks", "Cost", "DirectBannerID"]].copy()
    df_chart["QueryBoldShare"] = df_chart.QueryHighlighTitleBody / df_chart.QueryWordsCnt
    df_chart = df_chart[df_chart.Shows > 0]
    if df_chart.shape[0] == 0:
        df_chart = None

    df_chart.QueryBoldShare = (df_chart.QueryBoldShare * 100).round(-1).astype(int).values
    df_chart = (df_chart.groupby(["QueryBoldShare"], as_index=True)
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "DirectBannerID": lambda x: x.unique().size}))

    df_chart.rename(columns={
        "DirectBannerID": u"Кол-во объявлений",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы"},
        inplace=True)
    df_chart.index.names = [u"Подсветка"]
    df_chart = df_chart.loc[range(0, 110, 10)].fillna(pd.np.nan)

    if normalize:
        df_chart = df_chart.apply(lambda x: x / x.sum())
        # Все пороги из головы эксперта, можно менять
        total_shows = df.Shows.sum() * 1.0
        bad_shows = total_shows - (df.QueryHighlighTitleBody / df.QueryWordsCnt * df.Shows).sum()
        if bad_shows / total_shows > 0.5:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="01_highlight_hist_queries",
                title=u"Подсветка",
                snippet=u"Можно улучшить подсветку. Не менее половины показов приходится на фразы без полной подсветки или без подсветки совсем",
                importance_lvl=2)

            append_recommendation(df, Rec)

    df_chart = df_chart[[u"Кол-во объявлений", u"Показы", u"Клики", u"Расходы"]]

    if normalize:
        formats = {
            u"Кол-во объявлений": "percent",
            u"Показы": "percent",
            u"Клики": "percent",
            u"Расходы": "percent"}
    else:
        formats = {
            u"Кол-во объявлений": "int",
            u"Показы": "int",
            u"Клики": "int",
            u"Расходы": "float"}

    setattr(df_chart, "formats", formats)

    return df_chart


def plot_highlight_queries_hist(writer, sheet_name, df):
    """
    (ExcelWriter, str, DataFrame) -> ExcelWriter
    01_highlight_hist_queries
    """

    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "column", None, u"Подсветка поисковых запросов")

    chart.set_x_axis({"name_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "major_tick_mark": "none",
                      "minor_tick_mark": "none",
                      "line": {"visible": True, "width": 1, "color": "#000000"},
                      "major_gridlines": {"visible": False}
                      })

    chart.set_y_axis({"name": u"Доля от общего",
                      "name_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_format": "#0%",
                      "major_tick_mark": "none",
                      "minor_tick_mark": "none",
                      "line": {"none": True},
                      "major_gridlines": {"visible": True,
                                          "line": {"width": 1, "color": "#d9d9d9"}}
                      })

    cols = "BCDE"
    last_row = df.index.size + 1
    for n, col in enumerate(cols):
        chart_add_series(chart, n, {
            "values": u"='{0}'!{1}2:{1}{2}".format(sheet_name, col, last_row),
            "categories": u"='{}'!A2:A{}".format(sheet_name, last_row),
            "name": u"='{0}'!{1}1".format(sheet_name, col),
            "data_labels": {"value": False},
            "gap": 150,
            "overlap": -20
        })

    sheet.insert_chart('A%s' % (last_row + 2), chart)

    return writer


def table_top_intersection_by_queries(df, match_type, min_shows=50):
    """03_top_intersection_by_queries
    Note:
    https://yql.yandex-team.ru/Queries/595219793d483f59fea7d6f4
    """
    assert match_type in ("fuzzy", "exact"), "Unknown `match_type`: %s" % match_type
    df_copy = (df.groupby(
        ["Query", "GroupID", "PhraseText", "CampaignID", "CampaignName"], as_index=False)
        .agg({"Shows": sum, "Clicks": sum, "Cost": sum}))
    total_shows = df_copy.Shows.sum()
    #  self join
    df_j = pd.merge(df_copy, df_copy, on="Query", how="inner")
    # Только пары из разных групп
    df_j = df_j.loc[df_j.GroupID_x < df_j.GroupID_y]
    # Фильтрация в зависимости от параметров - дубли или все кроме дублей
    if match_type == "exact":
        df_j = df_j.loc[(df_j.PhraseText_x == df_j.PhraseText_y)]
    elif match_type == "fuzzy":
        df_j = df_j.loc[(df_j.PhraseText_x != df_j.PhraseText_y)]
    df_j["Count"] = 1
    df_j = df_j.groupby([
        "PhraseText_x", "PhraseText_y",
        "GroupID_x", "GroupID_y",
        "CampaignID_x", "CampaignID_y",
        "CampaignName_x", "CampaignName_y"],
        as_index=False).agg(
        {
            "Shows_x": sum,
            "Shows_y": sum,
            "Clicks_x": sum,
            "Clicks_y": sum,
            "Cost_x": sum,
            "Cost_y": sum,
            "Count": sum
        }
    )
    df_j = df_j.loc[(df_j.Shows_x + df_j.Shows_y) > min_shows]
    df_j = df_j.sort_values("Count", ascending=False).reset_index()
    df_j = df_j[[
        "PhraseText_x", "PhraseText_y",
        "GroupID_x", "GroupID_y",
        "CampaignID_x", "CampaignID_y",
        "CampaignName_x", "CampaignName_y",
        "Shows_x", "Shows_y",
        "Clicks_x", "Clicks_y",
        "Cost_x", "Cost_y",
        "Count"]]
    df_j.rename(columns={
        "PhraseText_x": u"Условие показа X",
        "PhraseText_y": u"Условие показа Y",
        "GroupID_x": u"Группа № X",
        "GroupID_y": u"Группа № Y",
        "CampaignID_x": u"Кампания X",
        "CampaignID_y": u"Кампания Y",
        "CampaignName_x": u"Кампания (имя) X",
        "CampaignName_y": u"Кампания (имя) Y",
        "Shows_x": u"Показы X",
        "Shows_y": u"Показы Y",
        "Clicks_x": u"Клики X",
        "Clicks_y": u"Клики Y",
        "Cost_x": u"Расходы X",
        "Cost_y": u"Расходы Y",
        "Count": u"Кол-во запросов"},
        inplace=True)
    intersected_shows = df_j.loc[:, u"Показы X"].sum() + df_j.loc[:, u"Показы Y"].sum()
    df_chart = df_j.head(EXCEL_MAX_ROWS)

    setattr(df_chart, "formats", {u"Условие показа X": "string",
                                  u"Условие показа Y": "string",
                                  u"Группа № X": "string",
                                  u"Группа № Y": "string",
                                  u"Кампания X": "string",
                                  u"Кампания Y": "string",
                                  u"Кампания (имя) X": "string",
                                  u"Кампания (имя) Y": "string",
                                  u"Показы X": "int",
                                  u"Показы Y": "int",
                                  u"Клики X": "int",
                                  u"Клики Y": "int",
                                  u"Расходы X": "float",
                                  u"Расходы Y": "float",
                                  u"Кол-во запросов": "int"})

    # Минимальный порог из головы эксперта, можно менять
    if (intersected_shows * 1. / total_shows) > 0.05 and match_type == "fuzzy":
        Rec = Recommendation(
            dataframe=df,
            internal_sheet_name="03_top_intersection_by_queries",
            title=u"Пересечения",
            snippet=u"В кампаниях могут быть пересечения",
            importance_lvl=2)

        append_recommendation(df, Rec)

    if (intersected_shows * 1. / total_shows) > 0.05 and match_type == "exact":
        Rec = Recommendation(
            dataframe=df,
            internal_sheet_name="03_top_doubles_by_queries",
            title=u"Дубли",
            snippet=u"В кампаниях могут быть дубли",
            importance_lvl=2)

        append_recommendation(df, Rec)

    return df_chart


def table_dssm_score_queries(df, normalize):
    """04_dssm_score_hist_queries"""

    df_chart = df[["DSSMScore", "Shows", "Clicks", "Cost", "DirectBannerID"]].copy()

    df_chart = df_chart[df_chart.Shows > 0]
    if df_chart.shape[0] == 0:
        df_chart = None

    df_chart.DSSMScore = (df_chart.DSSMScore * 100).round(-1).astype(int).values
    df_chart = (df_chart.groupby(["DSSMScore"], as_index=True)
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "DirectBannerID": lambda x: x.unique().size}))

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart.rename(columns={
        "DirectBannerID": u"Кол-во объявлений",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": "CPC",
        "CTR": "CTR"},
        inplace=True)
    df_chart.index.names = [u"Релевантность"]
    df_chart = df_chart.loc[range(0, 110, 10)].fillna(pd.np.nan)

    sum_cols = [u"Кол-во объявлений", u"Показы", u"Клики", u"Расходы"]
    max_cols = [u"CPC", u"CTR"]

    if normalize:
        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())

        # Все пороги из головы эксперта, можно менять
        total_shows = df.Shows.sum() * 1.0
        bad_shows = df.query("DSSMScore < 0.75").Shows.sum() * 1.0
        if bad_shows / total_shows > 0.5:
            Rec = Recommendation(
                dataframe=df,
                internal_sheet_name="04_dssm_score_hist_queries",
                title=u"Релевантность",
                snippet=u"Более половины покaзов может приходиться на нерелевантные запросы (нецелевые запросы, недостаток минус-фраз)",
                importance_lvl=2)

            append_recommendation(df, Rec)

        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

    df_chart = df_chart.loc[:, sum_cols + max_cols]

    if normalize:
        formats = {u"Кол-во объявлений": "percent",
                   u"Показы": "percent",
                   u"Клики": "percent",
                   u"Расходы": "percent",
                   u"CPC": "float",
                   u"CTR": "float"}
    else:
        formats = {u"Кол-во объявлений": "int",
                   u"Показы": "int",
                   u"Клики": "int",
                   u"Расходы": "float",
                   u"CPC": "float",
                   u"CTR": "percent"}

    setattr(df_chart, "formats", formats)

    return df_chart


def plot_dssm_score_queries_hist(writer, sheet_name, df):
    """
    (ExcelWriter, str, DataFrame) -> ExcelWriter
    04_dssm_score_hist_queries
    """

    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "column", None, u"Релевантность пар <запрос пользователя, текст баннера>")

    chart.set_x_axis({"name_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "major_tick_mark": "none",
                      "minor_tick_mark": "none",
                      "line": {"visible": True, "width": 1, "color": "#000000"},
                      "major_gridlines": {"visible": False}
                      })

    chart.set_y_axis({"name": u"Доля от общего",
                      "name_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_format": "#0%",
                      "major_tick_mark": "none",
                      "minor_tick_mark": "none",
                      "line": {"none": True},
                      "major_gridlines": {"visible": True,
                                          "line": {"width": 1, "color": "#d9d9d9"}}
                      })

    cols = "BCDE"
    last_row = df.index.size + 1
    for n, col in enumerate(cols):
        chart_add_series(chart, n, {
            "values": u"='{0}'!{1}2:{1}{2}".format(sheet_name, col, last_row),
            "categories": u"='{}'!A2:A{}".format(sheet_name, last_row),
            "name": u"='{0}'!{1}1".format(sheet_name, col),
            "data_labels": {"value": False},
            "gap": 150,
            "overlap": -20
        })

    sheet.insert_chart('A%s' % (last_row + 2), chart)

    return writer


def table_top_irrelevant_queries(df, score_from=.0, score_to=.3):
    """04_top_irrel_queries_x_y

    score_from < score <= score_to
    """

    if not (0 <= score_from <= 1):
        raise ValueError("Invalid value of `score_from`: %s" % score_from)
    if not (0 <= score_to <= 1):
        raise ValueError("Invalid value of `score_to`: %s" % score_to)

    df_chart = df[[
        "CampaignID", "CampaignName", "DirectBannerID", "Title", "Body", "Query",
        "PhraseText", "Shows", "Clicks", "Cost", "DSSMScore"]].copy()
    df_chart = df_chart[
        (df_chart.Shows > 0) &
        (df_chart.DSSMScore > score_from) &
        (df_chart.DSSMScore <= score_to)]

    if df_chart.shape[0] == 0:
        return None

    # Топ по количеству нерелевантных показов
    df_chart['ShowsIrrelevant'] = ((1 - df_chart['DSSMScore']) * df_chart['Shows']).astype(int)
    df_chart = df_chart.sort_values("ShowsIrrelevant", ascending=False).head(EXCEL_MAX_ROWS)
    df_chart['HelperURL'] = df_chart.apply(
        lambda row: get_cid_bid(row["CampaignID"], row["DirectBannerID"]),
        axis=1)

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "DirectBannerID": u"Баннер №",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "Query": u"Поисковый запрос",
        "PhraseText": u"Условие показа",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": u"CPC",
        "CTR": u"CTR",
        "DSSMScore": u"Релевантность",
        "HelperURL": u"Ссылка на баннер"},
        inplace=True)

    df_chart = df_chart.loc[:, [
        u"Кампания №", u"Название кампании", u"Баннер №", u"Заголовок", u"Объявление",
        u"Поисковый запрос", u"Условие показа",
        u"Показы", u"Клики", u"Расходы", u"CPC", u"CTR", u"Ссылка на баннер"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Баннер №": "string",
                                  u"Заголовок": "string",
                                  u"Объявление": "string",
                                  u"Поисковый запрос": "string",
                                  u"Условие показа": "string",
                                  u"Показы": "int",
                                  u"Клики": "int",
                                  u"Расходы": "float",
                                  u"CPC": "float",
                                  u"CTR": "percent",
                                  u"Ссылка на баннер": "string"})

    return df_chart


def translate_camp_type(cell):
    camp_types = {
        'TEXT_CAMPAIGN': u'Текстово-графические объявления',
        'MOBILE_APP_CAMPAIGN': u'Реклама мобильных приложений',
        'DYNAMIC_TEXT_CAMPAIGN': u'Динамические объявления'}
    return camp_types.get(cell, cell)


def table_top_irrelevant_campaigns(df):
    """04_top_irrel_campaigns"""

    df_chart = df[[
        "CampaignID", "CampaignType", "CampaignName", "GroupID", "Shows", "Clicks", "Cost", "DSSMScore"]].copy()
    df_chart = df_chart.loc[df_chart.Shows > 0, :]

    if df_chart.shape[0] == 0:
        return None

    # Топ по количеству нерелевантных показов
    df_chart['ShowsIrrelevant'] = ((1 - df_chart['DSSMScore']) * df_chart['Shows']).astype(int)
    df_chart = (df_chart
                .groupby(["CampaignID", "CampaignType", "CampaignName"], as_index=False)
                .agg({
                    "Shows": sum,
                    "ShowsIrrelevant": sum,
                    "Clicks": sum,
                    "Cost": sum,
                    "GroupID": lambda s: s.unique().size
                })
                .sort_values("ShowsIrrelevant", ascending=False)
                .head(EXCEL_MAX_ROWS))
    df_chart['HelperURL'] = df_chart.apply(lambda row: get_cid(row["CampaignID"]), axis=1)

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart["ShowsIrrelevantShare"] = df_chart.ShowsIrrelevant / df_chart.Shows
    df_chart["CampaignType"] = df_chart.CampaignType.apply(translate_camp_type)

    df_chart.rename(columns={
        "CampaignID": u"Кампания, №",
        "CampaignType": u"Кампания, тип",
        "CampaignName": u"Кампания, имя",
        "Shows": u"Показы",
        "ShowsIrrelevantShare": u"Доля нерелевантных показов",
        "GroupID": u"Кол-во групп объявлений",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": u"CPC",
        "CTR": u"CTR",
        "HelperURL": u"Ссылка на кампанию"},
        inplace=True)

    df_chart = df_chart.loc[:, [u"Кампания, №",
                                u"Кампания, тип",
                                u"Кампания, имя",
                                u"Показы",
                                u"Доля нерелевантных показов",
                                u"Кол-во групп объявлений",
                                u"Клики",
                                u"Расходы",
                                u"CPC",
                                u"CTR",
                                u"Ссылка на кампанию"]]

    setattr(df_chart, "formats", {u"Кампания, №": "string",
                                  u"Кампания, тип": "string",
                                  u"Кампания, имя": "string",
                                  u"Показы": "int",
                                  u"Доля нерелевантных показов": "percent",
                                  u"Кол-во групп объявлений": "int",
                                  u"Клики": "int",
                                  u"Расходы": "float",
                                  u"CPC": "float",
                                  u"CTR": "percent",
                                  u"Ссылка на кампанию": "string"})

    return df_chart


def table_eshows_share_queries(df, normalize):
    """04_eshows_share_queries"""

    df_chart = df[["eShowsShare", "Shows", "Clicks", "Cost", "DirectPhraseID"]].copy()

    df_chart = df_chart[df_chart.Shows > 0]
    if df_chart.shape[0] == 0:
        return None

    df_chart.eShowsShare = (df_chart.eShowsShare * 100).round(-1).astype(int).values
    df_chart = (df_chart.groupby(["eShowsShare"], as_index=True)
                .agg({"Shows": pd.np.sum,
                      "Clicks": pd.np.sum,
                      "Cost": pd.np.sum,
                      "DirectPhraseID": lambda x: x.unique().size}))

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart.rename(columns={
        "DirectPhraseID": u"Кол-во фраз",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": "CPC",
        "CTR": "CTR"},
        inplace=True)
    df_chart.index.names = [u"Полученные показы"]
    df_chart = df_chart.loc[range(0, 110, 10)].fillna(pd.np.nan)

    sum_cols = [u"Кол-во фраз", u"Показы", u"Клики", u"Расходы"]
    max_cols = [u"CPC", u"CTR"]

    if normalize:
        df_chart.loc[:, sum_cols] = df_chart.loc[:, sum_cols].apply(lambda x: x / x.sum())
        df_chart.loc[:, max_cols] = df_chart.loc[:, max_cols].apply(lambda x: x / x.max())

    df_chart = df_chart.loc[:, sum_cols + max_cols]

    if normalize:
        formats = {u"Кол-во фраз": "percent",
                   u"Показы": "percent",
                   u"Клики": "percent",
                   u"Расходы": "percent",
                   u"CPC": "float",
                   u"CTR": "float"}
    else:
        formats = {u"Кол-во фраз": "int",
                   u"Показы": "int",
                   u"Клики": "int",
                   u"Расходы": "float",
                   u"CPC": "float",
                   u"CTR": "percent"}

    setattr(df_chart, "formats", formats)

    return df_chart


def plot_eshows_share_queries_hist(writer, sheet_name, df):
    """
    (ExcelWriter, str, DataFrame) -> ExcelWriter
    04_eshows_share_queries
    """

    sheet = writer.book.get_worksheet_by_name(sheet_name)
    chart = Chart(writer.book, "column", None, u"Доля полученных показов")

    chart.set_x_axis({"name_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "major_tick_mark": "none",
                      "minor_tick_mark": "none",
                      "line": {"visible": True, "width": 1, "color": "#000000"},
                      "major_gridlines": {"visible": False}
                      })

    chart.set_y_axis({"name": u"Доля от общего",
                      "name_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_font": {"size": FONT_SIZE, "bold": False, "name": FONT_NAME},
                      "num_format": "#0%",
                      "major_tick_mark": "none",
                      "minor_tick_mark": "none",
                      "line": {"none": True},
                      "major_gridlines": {"visible": True,
                                          "line": {"width": 1, "color": "#d9d9d9"}}
                      })

    cols = "BCDE"
    last_row = df.index.size + 1
    for n, col in enumerate(cols):
        chart_add_series(chart, n, {
            'values': u"='{0}'!{1}2:{1}{2}".format(sheet_name, col, last_row),
            "categories": u"='{}'!A2:A{}".format(sheet_name, last_row),
            "name": u"='{0}'!{1}1".format(sheet_name, col),
            "data_labels": {"value": False},
            "gap": 150,
            "overlap": -20
        })

    sheet.insert_chart('A%s' % (last_row + 2), chart)

    return writer


def table_top_eshows_share_queries(df, score_from=.0, score_to=1.0):
    """04_top_eshows_share_queries

    score_from < score <= score_to
    """

    df_chart = df[[
        "CampaignID", "CampaignName", "DirectBannerID", "Title", "Body", "Query",
        "PhraseText", "Shows", "Clicks", "Cost", "eShowsShare", "NewShows"]].copy()
    df_chart = df_chart[
        (df_chart.Shows > 0) &
        (df_chart.eShowsShare > score_from) &
        (df_chart.eShowsShare <= score_to)]

    if df_chart.shape[0] == 0:
        return None

    # Топ по количеству потенциальных кликов
    df_chart = (
        df_chart
        .sort_values("NewShows", ascending=False)
        .query("NewShows >= 10")
        .head(EXCEL_MAX_ROWS))
    if df_chart.shape[0] == 0:
        return None
    df_chart['HelperURL'] = df_chart.apply(
        lambda row: get_cid_bid(row["CampaignID"], row["DirectBannerID"]),
        axis=1)

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart.rename(columns={
        "CampaignID": u"Кампания №",
        "CampaignName": u"Название кампании",
        "DirectBannerID": u"Баннер №",
        "Title": u"Заголовок",
        "Body": u"Объявление",
        "Query": u"Поисковый запрос",
        "PhraseText": u"Условие показа",
        "Shows": u"Показы",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": u"CPC",
        "CTR": u"CTR",
        "eShowsShare": u"Доля полученных показов",
        "NewShows": u"Новые eShows",
        "HelperURL": u"Ссылка на баннер"},
        inplace=True)

    df_chart = df_chart.loc[:, [
        u"Кампания №", u"Название кампании", u"Баннер №", u"Заголовок", u"Объявление",
        u"Поисковый запрос", u"Условие показа",
        u"Доля полученных показов", u"Новые eShows",
        u"Показы", u"Клики", u"Расходы", u"CPC", u"CTR", u"Ссылка на баннер"]]

    setattr(df_chart, "formats", {u"Кампания №": "string",
                                  u"Название кампании": "string",
                                  u"Баннер №": "string",
                                  u"Заголовок": "string",
                                  u"Объявление": "string",
                                  u"Поисковый запрос": "string",
                                  u"Условие показа": "string",
                                  u"Показы": "int",
                                  u"Доля полученных показов": "float",
                                  u"Клики": "int",
                                  u"Новые eShows": "float",
                                  u"Расходы": "float",
                                  u"CPC": "float",
                                  u"CTR": "percent",
                                  u"Ссылка на баннер": "string"})

    return df_chart


def table_top_eshows_share_campaigns(df):
    """04_top_eshows_share_campaigns"""

    df_chart = df[[
        "CampaignID", "CampaignType", "CampaignName", "Shows", "Clicks", "Cost", "NewShows", "eShows", "MaxX"]].copy()
    df_chart = df_chart.loc[df_chart.Shows > 0, :]

    if df_chart.shape[0] == 0:
        return None

    # Топ по количеству новых eShows
    df_chart = (df_chart
                .groupby(["CampaignID", "CampaignType", "CampaignName"], as_index=False)
                .agg({
                    "Shows": sum,
                    "eShows": sum,
                    "MaxX": sum,
                    "NewShows": sum,
                    "Clicks": sum,
                    "Cost": sum,
                })
                .assign(
                    eShowsShare=lambda xf: xf.eShows / xf.MaxX
                )
                .sort_values("NewShows", ascending=False)
                .head(EXCEL_MAX_ROWS))
    df_chart['HelperURL'] = df_chart.apply(lambda row: get_cid(row["CampaignID"]), axis=1)

    add_cpc_total(df_chart)
    add_ctr_total(df_chart)

    df_chart["CampaignType"] = df_chart.CampaignType.apply(translate_camp_type)

    df_chart.rename(columns={
        "CampaignID": u"Кампания, №",
        "CampaignType": u"Кампания, тип",
        "CampaignName": u"Кампания, имя",
        "Shows": u"Показы",
        "eShowsShare": u"Доля полученных показов",
        "NewShows": u"Новые eShows",
        "Clicks": u"Клики",
        "Cost": u"Расходы",
        "CPC": u"CPC",
        "CTR": u"CTR",
        "HelperURL": u"Ссылка на кампанию"},
        inplace=True)

    df_chart = df_chart.loc[:, [u"Кампания, №",
                                u"Кампания, тип",
                                u"Кампания, имя",
                                u"Показы",
                                u"Доля полученных показов",
                                u"Клики",
                                u"Новые eShows",
                                u"Расходы",
                                u"CPC",
                                u"CTR",
                                u"Ссылка на кампанию"]]

    setattr(df_chart, "formats", {u"Кампания, №": "string",
                                  u"Кампания, тип": "string",
                                  u"Кампания, имя": "string",
                                  u"Показы": "int",
                                  u"Доля полученных показов": "percent",
                                  u"Клики": "int",
                                  u"Новые eShows": "float",
                                  u"Расходы": "float",
                                  u"CPC": "float",
                                  u"CTR": "percent",
                                  u"Ссылка на кампанию": "string"})

    return df_chart
