'use strict';

const fs = require('fs');

const { createCanvas, Image } = require('canvas');

class basePainter {
    /**
     * Базовый конструктор. Этот класс занимается рисованием шэрингов
     * @param {object} config - данные для рисования шэринга
     */
    constructor(config) {
        this._config = Object.assign({}, config);
    }

    /**
     * Создаёт базовый канвас фиксированного размера и инициализирует его контекст
     * @protected
     */
    _createCanvas() {
        this._canvas = createCanvas(this._config.canvas.width, this._config.canvas.height);
        this._ctx = this._canvas.getContext('2d');
    }

    /**
     * Создаёт прямоугольник с размером из конфига
     * @param {object} options
     * @protected
     */
    _fillRect(options) {
        options = options || {};

        this._ctx.fillStyle = options.fillStyle || '#000';

        this._ctx.fillRect(0, 0, this._config.canvas.width, this._config.canvas.height);
    }

    /**
     * Создаёт фон фиксрованного размера и цвета
     * @protected
     */
    _fillBackground() {
        this._fillRect({
            fillStyle: this._config.background.color
        });
    }

    /**
     * Отрисовывает файл с лого по заданным параметрам
     * @param {number} offsetX
     * @param {number} offsetY
     * @param {string} imagePath
     * @param {boolean} needScale
     * @protected
     */
    _fillImage({ offsetX = 0, offsetY = 0, imagePath, needScale }) {
        const image = new Image();
        image.src = fs.readFileSync(imagePath);

        let imgWidth = image.width;
        let imgHeight = image.height;

        if (needScale) {
            const scaleRatio = this._getScaleRatio(image);

            imgWidth *= scaleRatio;
            imgHeight *= scaleRatio;

            // Т.к мы изменили размер изображения, то нам нужно получить смещения, чтобы отрендерить картинку по центру
            offsetX += (this._config.canvas.width - imgWidth) / 2;
            offsetY += (this._config.canvas.height - imgHeight) / 2;
        }

        this._ctx.drawImage(image, offsetX, offsetY, imgWidth, imgHeight);
    }

    /**
     * Подбирает коэффициент для масштабирования по ширине и высоте так, чтобы итоговый размер был хорошим
     * @param {Image} image
     * @returns {Number}
     * @private
     */
    _getScaleRatio(image) {
        let scaleRatio = this._config.canvas.width / image.width;
        const newHeight = scaleRatio * image.height;

        if (newHeight < this._config.canvas.height) {
            scaleRatio *= this._config.canvas.height / newHeight;
        }

        return scaleRatio;
    }

    /**
     * Пишет текст на картинке
     * @param {number} offsetX
     * @param {number} offsetY
     * @param {string} text
     * @param {string} color
     * @param {string} font
     * @param {number} fontSize
     * @param {number} lineHeight
     * @protected
     */
    _fillText({ offsetX, offsetY, text, color, font, fontSize, lineHeight }) {
        this._ctx.fillStyle = color;
        this._ctx.font = `${fontSize}px ${font}`;

        const lines = text.split('\n');

        lines.forEach(line => {
            this._ctx.fillText(line, offsetX, offsetY);
            offsetY += lineHeight;
        });
    }

    /**
     * Возвращает base64 строку в формате image/png
     * @returns {String}
     */
    get base64() {
        return this._canvas.toDataURL();
    }
}

module.exports = basePainter;
