const config = require('cfg');
const http = require('http');
const {format} = require('url');
const {Image} = require('canvas');

const PainterBase = require('./painter');

class OgPainter extends PainterBase {
    constructor(config) {
        super(config);

        this._config.fonts.forEach(font => {
            font.faces.forEach(face => {
                this._registerFont(font.family, face, font.path);
            });
        });
    }

    /**
     * Затемняет фон
     * @param opacity
     * @private
     */
    _fillVeil(opacity = 0) {
        if (typeof opacity === 'string') {
            opacity = parseInt(opacity, 10);
        }

        opacity = opacity > 1 ? opacity / 100 : opacity;

        this._fillRect({
            fillStyle: this._config.veil.color.replace('$', opacity.toString())
        });
    }

    _fillTextarea(params) {
        this._fillText({
            offsetX: this._config.text.offsetX,
            offsetY: this._config.text.offsetY,
            text: params.text,
            color: params.isInverted ? this._config.text.invertedColor : this._config.text.color,
            font: this._config.text.font,
            lineHeight: this._config.text.lineHeight
        });
    }

    _fillUrl(params) {
        this._fillText({
            offsetX: this._config.url.offsetX,
            offsetY: this._config.url.offsetY,
            text: params.url,
            color: params.isInverted ? this._config.url.invertedColor : this._config.url.color,
            font: this._config.url.font,
            lineHeight: 0
        });
    }

    _fetchDynamicLogo(params) {
        const color = config.logoaas.color;
        const invertedColor = config.logoaas.invertedColor;
        const ext = config.logoaas.format;

        let url = config.logoaas.url;

        url = format({
            protocol: url.protocol,
            host: url.host,
            pathname: encodeURI(`${url.pathname}/${params.textLogo}.${ext}`),
            query: {
                ...url.query,
                color: params.isInverted ? invertedColor : color
            }
        });

        return new Promise((resolve, reject) => {
            try {
                http.get(url, res => {
                    let buf = '';

                    res.setEncoding('binary');

                    res.on('data', function (chunk) {
                        buf += chunk;
                    });

                    res.on('end', function () {
                        resolve(Buffer.from(buf, 'binary'));
                    });
                });
            } catch (err) {
                reject(err);
            }
        });
    }

    _fillDynamicLogo(image) {
        const dynamicOffset = this._config.canvas.width - image.width - this._config.logo.rightOffsetX;

        this._fillImage({
            offsetX: dynamicOffset,
            offsetY: this._config.logo.offsetY,
            image
        });
    }

    /**
     * Генерирует og-image
     * @param params
     * returns dataURL
     */
    async generateOg(params) {
        this._createCanvas();

        if (params.backgroundImage) {
            const image = new Image();

            image.src = params.backgroundImage;

            this._fillImage({
                image,
                needScale: true
            });
        } else {
            this._fillBackground();
        }

        if (params.opacity) {
            this._fillVeil(params.opacity);
        }

        if (params.textLogo) {
            const image = new Image();

            image.src = await this._fetchDynamicLogo(params);

            this._fillDynamicLogo(image);
        }

        if (params.text) {
            this._fillTextarea(params);
        }

        if (params.url) {
            this._fillUrl(params);
        }

        return this.base64;
    }
}

module.exports = OgPainter;
