const path = require('path');

const {createCanvas, registerFont} = require('canvas');

class basePainter {
    /**
     * Базовый конструктор. Этот класс занимается рисованием og-картинок
     * @param {object} config - данные для рисования og-картинки
     */
    constructor(config) {
        this._config = Object.assign({}, config);
    }

    /**
     * Создаёт базовый канвас фиксированного размера и инициализирует его контекст
     * @private
     */
    _createCanvas() {
        this._canvas = createCanvas(this._config.canvas.width, this._config.canvas.height);
        this._ctx = this._canvas.getContext('2d');
    }

    /**
     * Создаёт прямоугольник с размером из конфига
     * @param fillStyle
     * @private
     */
    _fillRect({fillStyle = '#000'}) {
        this._ctx.fillStyle = fillStyle;

        this._ctx.fillRect(0, 0, this._config.canvas.width, this._config.canvas.height);
    }

    /**
     * Создаёт фон фиксрованного размера и цвета
     * @private
     */
    _fillBackground() {
        this._fillRect({
            fillStyle: this._config.background.color
        });
    }

    /**
     * Отрисовывает файл с лого по заданным параметрам
     * @param offsetX
     * @param offsetY
     * @param img
     * @param needScale
     * @private
     */
    _fillImage({offsetX = 0, offsetY = 0, image, needScale}) {
        let imgWidth = image.width;
        let imgHeight = image.height;

        if (needScale) {
            const scaleRatio = this._getScaleRatio(image);

            imgWidth *= scaleRatio;
            imgHeight *= scaleRatio;

            // Т.к мы изменили размер изображения, то нам нужно получить смещения, чтобы отрендерить картинку по центру
            offsetX += (this._config.canvas.width - imgWidth) / 2;
            offsetY += (this._config.canvas.height - imgHeight) / 2;
        }

        this._ctx.drawImage(image, offsetX, offsetY, imgWidth, imgHeight);
    }

    /**
     * Подбирает коэффициент для масштабирование по ширине и высоте так, чтобы итоговый размер был хорошим:
     * @param image - Canvas.Image object
     * @returns {Number}
     * @private
     */
    _getScaleRatio(image) {
        let scaleRatio = this._config.canvas.width / image.width;
        const newHeight = scaleRatio * image.height;

        if (newHeight < this._config.canvas.height) {
            scaleRatio *= this._config.canvas.height / newHeight;
        }

        return scaleRatio;
    }

    /**
     * Регистрация шрифта
     * @param family
     * @param face
     * @param fontPath
     * @private
     */
    _registerFont(family, face, fontPath) {
        registerFont(
            path.join(__dirname, fontPath, face.fileName),
            {
                family: family,
                weight: face.weight,
                style: face.style
            }
        );
    }

    /**
     * Пишет текст на картинке
     * @param offsetX
     * @param offsetY
     * @param text
     * @param color
     * @param font
     * @param lineHeight
     * @private
     */
    _fillText({offsetX, offsetY, text, color, font, lineHeight}) {
        this._ctx.fillStyle = color;
        this._ctx.font = `${font.weight} ${font.style} ${font.size}px ${font.family}`;

        const lines = text.split('\n');

        lines.forEach(line => {
            this._ctx.fillText(line, offsetX, offsetY);
            offsetY += lineHeight;
        });
    }

    /**
     * Возвращает base64 строку в формате image/png
     * @returns {String}
     */
    get base64() {
        return this._canvas.toDataURL();
    }
}

module.exports = basePainter;
