import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {observer, inject} from 'mobx-react';

import ControlPanel from '../../components/control-panel';
import Path from '../../components/path';
import ExplorerItem from '../../components/explorer-item';

import './index.css';

@inject('s3PageStore')
@observer
export default class Explorer extends Component {
    static contextTypes = {
        s3StorageExternal: PropTypes.string.isRequired
    };

    static propTypes = {
        s3PageStore: PropTypes.object.isRequired
    };

    removeSymbolsFromDirName = dirName => {
        return dirName
            .replace(new RegExp('#', 'g'), '')
            .replace(new RegExp(this.props.s3PageStore.delimiter, 'g'), '')
            .replace(new RegExp('\\\\', 'g'), '')
            .replace(new RegExp(' {2,}', 'g'), ' ')
            .trim()
            .replace(new RegExp(' ', 'g'), '_');
    };

    handleCreateBucket = () => {
        const bucketName = prompt("Введите название bucket'а");

        if (bucketName) {
            const formData = new FormData();

            formData.append('bucket', bucketName);

            this.props.s3PageStore.createBucket(formData);
        }
    };

    handleRemoveDir = e => {
        e.stopPropagation();

        const s3Dir = e.currentTarget.parentNode.parentNode; // TODO: поправить
        const s3DirName = s3Dir.getAttribute('data-name');
        const s3DirPrefix = s3Dir.getAttribute('data-prefix');

        if (confirm(`Удалить папку ${s3DirName}?`)) {
            const formData = new FormData();

            formData.append('bucket', this.props.s3PageStore.currentBucket);

            this.props.s3PageStore.deleteObjects({
                bucket: this.props.s3PageStore.currentBucket,
                delimiter: this.props.s3PageStore.delimiter,
                prefix: s3DirPrefix
            }, formData);
        }
    };

    handleClick = e => {
        const type = e.currentTarget.getAttribute('data-type');

        if (type === 'bucket') {
            const name = e.currentTarget.getAttribute('data-name');

            this.props.s3PageStore.listObjects({
                bucket: name,
                delimiter: this.props.s3PageStore.delimiter
            });
        }

        if (type === 'dir') {
            const prefix = e.currentTarget.getAttribute('data-prefix');

            this.props.s3PageStore.listObjects({
                bucket: this.props.s3PageStore.currentBucket,
                delimiter: this.props.s3PageStore.delimiter,
                prefix
            });
        }

        if (type === 'file') {
            console.info('file!');
        }
    };

    handleClickBack = () => {
        this.props.s3PageStore.history.pop(); // извлекаем текущий

        if (this.props.s3PageStore.history.length) {
            this.props.s3PageStore.listObjects(this.props.s3PageStore.history.pop()); // прыгаем на предыдущий
        } else {
            this.props.s3PageStore.listBuckets();
        }
    };

    handleFileChange = event => {
        const [file] = event.target.files;

        if (file) {
            const formData = new FormData();

            const key = this.props.s3PageStore.currentPrefix ? `${this.props.s3PageStore.currentPrefix}${file.name}` : file.name;

            formData.append('bucket', this.props.s3PageStore.currentBucket);
            formData.append('key', key);
            formData.append('file', file);

            this.props.s3PageStore.putObject(formData);
        }
    };

    handleRemoveFile = e => {
        e.stopPropagation();

        const s3Item = e.currentTarget.parentNode.parentNode; // TODO: поправить
        const s3ItemName = s3Item.getAttribute('data-name');

        if (confirm(`Удалить файл ${s3ItemName}?`)) {
            const formData = new FormData();

            const key = this.props.s3PageStore.currentPrefix ? `${this.props.s3PageStore.currentPrefix}${s3ItemName}` : s3ItemName;

            formData.append('bucket', this.props.s3PageStore.currentBucket);
            formData.append('key', key);

            this.props.s3PageStore.deleteObject(formData);
        }
    };

    handleRefresh = () => {
        this.props.s3PageStore.listObjects({
            bucket: this.props.s3PageStore.currentBucket,
            delimiter: this.props.s3PageStore.delimiter,
            prefix: this.props.s3PageStore.currentPrefix
        }, true);
    };

    handleCreateDir = e => {
        e.stopPropagation();

        let dirName = prompt('Введите название папки');

        if (dirName) {
            const formData = new FormData();

            dirName = this.removeSymbolsFromDirName(dirName);

            const key = this.props.s3PageStore.currentPrefix ? `${this.props.s3PageStore.currentPrefix}${dirName}` : dirName;

            formData.append('bucket', this.props.s3PageStore.currentBucket);
            formData.append('key', `${key}${this.props.s3PageStore.delimiter}`);

            this.props.s3PageStore.putObject(formData);
        }
    };

    handleChangeCredentials = () => {
        if (confirm('Вы действительно хотите сменить реквизиты?')) {
            this.props.s3PageStore.resetStore();
        }
    };

    // TODO: возвращать массив, когда вернёмся на 16-й React
    render() {
        return (
            <section>
                <ControlPanel
                    key={Math.random()}
                    onClickCreateBucket={this.handleCreateBucket}
                    onFileChange={this.handleFileChange}
                    onClickRefresh={this.handleRefresh}
                    onClickCreateDir={this.handleCreateDir}
                    onClickChangeCredentials={this.handleChangeCredentials}
                />
                <Path key="path"/>
                <ul key="explorer" className="explorer">
                    {
                        this.props.s3PageStore.currentDir.map(item => {
                            return (
                                <ExplorerItem
                                    key={item.name}
                                    item={item}
                                    onClick={this.handleClick}
                                    onClickBack={this.handleClickBack}
                                    downloadLink={this.props.s3PageStore.getDownloadLink(this.context.s3StorageExternal, item.name)}
                                    onClickRemoveDir={this.handleRemoveDir}
                                    onClickRemoveFile={this.handleRemoveFile}
                                />
                            );
                        })
                    }
                </ul>
            </section>
        );
    }
}
