import {observable, computed, action, runInAction, reaction} from 'mobx';

import buildQueryParams from './../../utils/build-query-params';
import errorHandler from './../../utils/error-handler';

import Bucket from './../models/bucket';
import BucketItemDir from './../models/bucketItemDir';
import BucketItemFile from './../models/bucketItemFile';

class S3PageStore {
    delimiter = '/';
    history = [];

    @observable errorMessage = null;

    @observable _accessKeyId = '';
    @observable _secretAccessKey = '';

    @observable isTryCreateBucket = false;

    @observable _isTryLogIn = false;
    @observable isLogIn = false;

    @observable currentBucket = null;
    @observable currentPrefix = null;
    @observable currentDir = [];
    @observable isUpdating = false;
    @observable isLoadingObject = false;

    @action('reset store')
    resetStore() {
        this.history = [];

        this.errorMessage = null;

        this.isTryCreateBucket = false;

        this.isTryLogIn = false;
        this.isLogIn = false;

        this.currentBucket = null;
        this.currentPrefix = null;
        this.currentDir = [];
        this.isUpdating = false;
        this.isLoadingObject = false;
    }

    get accessKeyId() {
        return this._accessKeyId;
    }

    @action('set accessKeyId')
    _setAccessKeyId(value) {
        this._accessKeyId = value;
    }

    set accessKeyId(value) {
        this._setAccessKeyId(value);
    }

    get secretAccessKey() {
        return this._secretAccessKey;
    }

    @action('set secretAccessKey')
    _setSecretAccessKey(value) {
        this._secretAccessKey = value;
    }

    set secretAccessKey(value) {
        this._setSecretAccessKey(value);
    }

    get isTryLogIn() {
        return this._isTryLogIn;
    }

    @action('set isTryLogIn')
    _setIsTryLogIn(value) {
        this._isTryLogIn = value;
    }

    set isTryLogIn(value) {
        this._setIsTryLogIn(value);
    }

    @computed
    get credentialHeaders() {
        return new Headers({
            'Access-Key-Id': this.accessKeyId,
            'Secret-Access-Key': this.secretAccessKey
        });
    }

    @computed
    get currentPrefixLen() {
        return this.currentPrefix ? this.currentPrefix.length : 0;
    }

    @action('list buckets')
    listBuckets() {
        fetch('s3-api/list-buckets', {
            headers: this.credentialHeaders,
            credentials: 'include'
        })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(json => {
                this.errorMessage = null;
                this.isTryLogIn = false;
                this.isLogIn = true;

                this.currentBucket = null;
                this.currentPrefix = null;
                this.currentDir = json
                    .Buckets
                    .map(bucket => new Bucket(bucket))
                    .sort((a, b) => a.name.localeCompare(b.name));
            })
            .catch(async err => {
                err = await errorHandler(err);

                console.error(err);

                runInAction(() => {
                    this.isTryLogIn = false;

                    this.errorMessage = err;
                });
            });
    }

    @action('create bucket')
    createBucket(formData) {
        this.isTryCreateBucket = true;

        fetch('s3-api/create-bucket', {
            method: 'POST',
            headers: this.credentialHeaders,
            credentials: 'include',
            body: formData
        })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(() => {
                this.listBuckets();

                this.isTryCreateBucket = false;
            })
            .catch(async err => {
                err = await errorHandler(err);

                console.error(err);

                runInAction(() => {
                    this.isTryCreateBucket = false;
                });
            });
    }

    @action('list objects')
    listObjects(queryParams, refresh = false) {
        this.isUpdating = true;

        fetch(`s3-api/list-objects?${buildQueryParams(queryParams)}`, {
            headers: this.credentialHeaders,
            credentials: 'include'
        })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(json => {
                this.currentBucket = queryParams.bucket;
                this.currentPrefix = queryParams.prefix;

                if (!refresh) {
                    this.history.push(queryParams);

                    console.info(this.history);
                }

                this.currentDir = [
                    new BucketItemDir({
                        Prefix: '..',
                        LastModified: null
                    }, 'dir-up'),
                    ...json
                        .CommonPrefixes
                        .map(item => new BucketItemDir(item))
                        .sort((a, b) => a.name.localeCompare(b.name)),
                    ...json
                        .Contents
                        .map(item => item.Key === json.Prefix ? null : new BucketItemFile(item)) // если создали папку
                        .filter(item => item ? item : false)
                        .sort((a, b) => a.name.localeCompare(b.name))
                ];

                this.isUpdating = false;
            })
            .catch(async err => {
                err = await errorHandler(err);

                console.error(err);

                runInAction(() => {
                    this.isUpdating = false;
                });
            });
    }

    getDownloadLink(storage, itemName) {
        const bucket = this.currentBucket;
        const prefix = this.currentPrefix || '';

        return `${storage}/${bucket}/${prefix}${encodeURIComponent(itemName)}`;
    }

    @action('put object')
    putObject(formData) {
        this.isLoadingObject = true;

        fetch('s3-api/put-object', {
            method: 'PUT',
            headers: this.credentialHeaders,
            credentials: 'include',
            body: formData
        })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(() => {
                this.listObjects({
                    bucket: this.currentBucket,
                    delimiter: this.delimiter,
                    prefix: this.currentPrefix
                }, true);

                this.isLoadingObject = false;
            })
            .catch(async err => {
                err = await errorHandler(err);

                console.error(err);

                runInAction(() => {
                    this.isLoadingObject = false;
                });
            });
    }

    @action('delete object')
    deleteObject(formData) {
        fetch('s3-api/delete-object', {
            method: 'DELETE',
            headers: this.credentialHeaders,
            credentials: 'include',
            body: formData
        })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(() => {
                this.listObjects({
                    bucket: this.currentBucket,
                    delimiter: this.delimiter,
                    prefix: this.currentPrefix
                }, true);
            })
            .catch(async err => {
                err = await errorHandler(err);

                console.error(err);
            });
    }

    /**
     * Функция удаляет все объекты в указанной папке.
     * Сначала мы делаем запрос для получения всех ключей в папке,
     * а вторым запросом отправляем эти ключи в ручку на массовое удаление.
     * @param {Object} queryParams
     * @param {FormData} formData
     */
    @action('delete objects')
    deleteObjects(queryParams, formData) {
        fetch(`s3-api/list-objects?${buildQueryParams(queryParams)}`, {
            headers: this.credentialHeaders,
            credentials: 'include'
        })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(json => {
                const keys = json
                    .Contents
                    .map(item => item.Key);

                formData.append('keys', JSON.stringify(keys));

                return fetch('s3-api/delete-objects', {
                    method: 'DELETE',
                    headers: this.credentialHeaders,
                    body: formData
                });
            })
            .then(res => res.status === 200 ? Promise.resolve(res) : Promise.reject(res))
            .then(res => res.json())
            .then(() => {
                this.listObjects({
                    bucket: this.currentBucket,
                    delimiter: this.delimiter,
                    prefix: this.currentPrefix
                }, true);
            })
            .catch(async err => {
                err = await errorHandler(err);

                console.error(err);
            });
    }
}

const s3PageStore = new S3PageStore();

reaction(
    () => s3PageStore.accessKeyId,
    accessKeyId => accessKeyId ? localStorage.setItem('accessKeyId', accessKeyId) : null
);

reaction(
    () => s3PageStore.secretAccessKey,
    secretAccessKey => secretAccessKey ? localStorage.setItem('secretAccessKey', secretAccessKey) : null
);

export default s3PageStore;
