import React from 'react';
import {renderToString} from 'react-dom/server';
import Helmet from 'react-helmet';
import htmlescape from 'htmlescape';

import getInitialState from './mobx/state';

const bundles = {
    index: require('./bundles/index/server').default
};

function renderDocument(helmet, appHtml, appState, appData) {
    return `<!doctype html>
<html ${helmet.htmlAttributes.toString()}>
    <head>
        ${helmet.title.toString()}
        ${helmet.meta.toString()}
        ${helmet.link.toString()}
        ${appHtml ? `<link rel="stylesheet" href="${appData.staticHost}/build/${appData.bundle}.build.css?${appData.appVersion}"/>` : ''}
        ${helmet.script.toString()}
    </head>
    <body ${helmet.bodyAttributes.toString()}>
        <div id="mount">${appHtml}</div>
        <script src="${appData.staticHost}/build/vendor.js?${appData.appVersion}"></script>
        <script src="${appData.staticHost}/build/${appData.bundle}.build.js?${appData.appVersion}"></script>
        <script nonce="${appData.nonce}">__init__.default(${htmlescape(appState)}, ${htmlescape(appData)}, ${!appHtml});</script>
    </body>
</html>`;
}

/**
 * Server-side render
 * @param {string}  bundleName Name of bundle to render
 * @param {string}  location Location for router (e.g. req.url)
 * @param {*}       appData Data passed to application
 * @param {boolean} [hotRender]
 * @returns {Promise}
 */
export default async function ssr(bundleName, location, appData, hotRender) {
    let appState = await getInitialState();

    const result = {};
    let appHtml = '';

    // server-side rendering if not hot render
    if (!hotRender) {
        if (!bundles[bundleName]) {
            throw new Error(`Bundle ${bundleName} not found`);
        }

        // create empty react-router context
        const context = {};

        // get requested bundle
        const Bundle = bundles[bundleName];

        // server-side rendering
        appHtml = renderToString(
            <Bundle
                appState={appState}
                appData={appData}
                location={location}
                context={context}
            />
        );

        result.redirectUrl = context.url;
    }

    // get helmet content
    const helmet = Helmet.renderStatic();

    result.html = renderDocument(helmet, appHtml, appState, appData);

    return result;
}
