// ADVDEV-1059: Скопировать все ноды из /products в /products-display
const _ = require('lodash');

const BunkerApi = require('../server/lib/bunker-api');

const bunkerApi = new BunkerApi({
    cookie: process.env.BUNKER_COOKIE
});

const pathRegex = /^\/adv-www\/sources\/(.+?)\/products\//;

/**
 * Загружает ноду в бункер.
 * 0 - нода загружена без ошибок
 * 1 - нода загружена без схемы
 * 2 - нода не загужена
 * @param {String} newPath
 * @param {Object} body
 * @param {String?} mime
 * @returns {Promise.<0 | 1 | 2>}
 */
const tryStore = async (newPath, body, mime) => {
    try {
        await bunkerApi.store(newPath, body, mime);
    } catch (error) {
        console.error(error);

        console.log('Загружаем без схемы');

        try {
            await bunkerApi.store(newPath, body);
        } catch (errorMime) {
            console.error(errorMime);

            return Promise.resolve(2);
        }

        return Promise.resolve(1);
    }

    return Promise.resolve(0);
};

/* eslint-disable complexity */
const copyProducts = async (version, tld) => {
    const nodes = await bunkerApi.list(`/adv-www/sources/${tld}/products`, {
        version,
        recursive: true
    });

    const problemNodes = [];
    const noSchema = [];

    for (const node of nodes) {
        const path = _.get(node, 'fullName', '');
        const newPath = path.replace(pathRegex, `/adv-www/sources/${tld}/products-display/`);

        console.log(`${path} -> ${newPath}`);

        let body = null;
        let mime = null;

        try {
            const nodeData = await bunkerApi.get(path, { version });

            body = _.get(nodeData, 'body');
            mime = _.get(nodeData, 'mime');
        } catch (error) {
            console.error(error);
        }

        if (!body || !mime) {
            problemNodes.push(path);
            continue;
        }

        const storeResult = await tryStore(newPath, body, mime);

        switch (storeResult) {
            case 1:
                noSchema.push(newPath);
                break;
            case 2:
                problemNodes.push(newPath);
                break;
        }

        if (storeResult === 2 || version !== 'stable') {
            continue;
        }

        try {
            await bunkerApi.publish(newPath);
        } catch (error) {
            console.error(error);
            problemNodes.push(path);
        }
    }

    return {
        problem: problemNodes,
        noSchema
    };
};
/* eslint-enable complexity */

(async () => {
    console.time('Продукты скопированы');

    const tld = 'ru';

    const stable = await copyProducts('stable', tld);

    console.log('stable-версия продуктов скопирована и опубликована');

    const latest = await copyProducts('latest', tld);

    console.timeEnd('Продукты скопированы');

    console.log('Скопированы без схемы:');
    console.log('stable', JSON.stringify(stable.noSchema, null, 4));
    console.log('latest', JSON.stringify(latest.noSchema, null, 4));

    console.log('Рекомендуется проверить следующие ноды:');
    console.log('stable', JSON.stringify(stable.problem, null, 4));
    console.log('latest', JSON.stringify(latest.problem, null, 4));
})();
