'use strict';

const bodyParser = require('body-parser');
const config = require('yandex-cfg');
const cookieParser = require('cookie-parser');
const express = require('express');
const expressBlackbox = require('express-blackbox');
const expressDomainAccess = require('express-domain-access');
const expressHttpUatraits = require('@yandex-int/express-http-uatraits');
const expressRenewBbSession = require('express-renew-bb-session');
const expressResponse400To404 = require('express-response-400to404');
const expressSecretKey = require('express-secretkey');
const expressTld = require('express-tld');
const expressTvm = require('@yandex-int/express-tvm').default;
const expressXff = require('express-x-forwarded-for-fix');
const expressYaFrameguard = require('express-ya-frameguard');
const expressYandexCsp = require('express-yandex-csp');
const expressYandexUid = require('@yandex-int/express-yandexuid');
const helmet = require('helmet');

const backend = require('./middleware/backend');
const bunker = require('./middleware/bunker');
const i18n = require('./middleware/i18n');
const language = require('./middleware/language');
const geobaseMock = require('./middleware/geobase-mock');
const locals = require('./middleware/locals');
const reactRender = require('./middleware/reactRender');
const expressBundleResponse = require('./middleware/express-bundle-response');
const securityHeaders = require('./middleware/security-headers');
const redirects = require('./middleware/redirects');
const referrerPolicy = require('./middleware/referrer-policy');
const laasGeobase = require('./middleware/laas-geobase');
const logger = require('./middleware/logger');

const app = express();

app.use(logger());
app.use(bunker());
/* eslint-disable no-unused-vars, max-params */
app.use((err, req, res, next) => {
    req.logger.error(err, 'Bunker error');

    next();
});

if (config.app.logRequests) {
    require('debug-http')();
}

app.set('env', config.environment);
app.disable('x-powered-by');
app.disable('etag');
app.enable('trust proxy');

app.use(referrerPolicy());

app.use(bodyParser.json());

app.use(reactRender);

app.use(`${config.router.vda}/static`, express.static(config.static.dir, {
    fallthrough: false,
    maxAge: 365 * 24 * 60 * 60 * 1000
}));

app.use(`${config.router.vda}/static2/_`, express.static(`${config.static2.dir}/client/_`, {
    fallthrough: false,
    maxAge: 365 * 24 * 60 * 60 * 1000
}));

app.use(`${config.router.vda}/static2`, express.static(`${config.static2.dir}/client`, {
    fallthrough: false,
    maxAge: 365 * 24 * 60 * 60 * 1000
}));

app.use(cookieParser());
app.use(expressXff());
app.use(expressYandexCsp(config.csp));

app.use(helmet.noSniff());
app.use(helmet.xssFilter({ setOnOldIE: true }));

app.use(expressTld());
app.use(redirects);
app.use(expressDomainAccess(config.app.domains));
app.use(geobaseMock());
app.use(expressYaFrameguard());
app.use(expressHttpUatraits(config.uatraits));
app.use(expressTvm(config.tvm));
app.use(expressBlackbox(config.blackbox));
app.use(expressYandexUid());
app.use(expressRenewBbSession());
app.use(expressSecretKey());
app.use(expressBundleResponse(config.view));
app.use(backend);
app.use(securityHeaders);
app.use(laasGeobase);

app.use(language);
app.use(i18n());
app.use(locals);

require('./router/redirects')(app);
require('./router/www-router')(app);
require('./router/router')(app);

app.use(expressResponse400To404(true));

/* eslint-disable no-unused-vars, max-params */
app.use((err, req, res, next) => {
    req.logger.error(err);

    res.sendStatus(500);
});

module.exports = app;
