'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');
const got = require('got');

const Base = require('./base');
const { Parser } = require('xml2js');

/* eslint-disable valid-jsdoc */
const templateHandlers = {

    /**
     * {{ form: id }}
     */
    form: stringData => {
        const match = stringData.match(/^{{form:([^}]*?)}}$/);

        return match ? `<a href="${config.forms.url}${match[1]}">Форма</a>` : '';
    },

    /**
     * {{ pseudolink : text }}
     */
    pseudolink: stringData => {
        const match = stringData.match(/^{{pseudolink:'([^']*?)'}}$/);

        return match ? `<span>${match[1]}</span>` : '';
    },

    /**
     * {{ author : name : description : url }}
     */
    author: stringData => {
        const match = stringData.match(/^{{author:'([^']*?)':'([^']*?)':'([^']*?)'}}$/);

        if (!match) {
            return '';
        }

        return `<img src="${match[3]}" width="70" height="70"><p>${match[1]}</p><p>${match[2]}</p>`;
    },

    /**
     * {{ cut : title : text }}
     */
    cut: stringData => {
        const match = stringData.match(/^{{cut:[^:]*?:'([^']*?)'}}$/);

        return match && match[1];
    },

    /**
     * {{ button : size : theme : text : url : metrikaGoal }}
     */
    button: stringData => {
        const match = stringData.match(/^{{button:[^:]*?:[^:]*?:'([^']*?)':'([^']*?)'/);

        if (!match) {
            return '';
        }

        if (match[2]) {
            return `<a href="${match[2]}">${match[1]}</a>`;
        }

        return `<button>${match[1]}</button>`;
    }
};
/* eslint-enable valid-jsdoc */

/**
 * Базовый класс для работы с RSS блогов
 */
class Rss extends Base {

    _getBlogsRss() {
        const blogId = config.blogs.news[this._req.tld] || config.blogs.news.default;
        const url = `${config.blogs.rss}/${blogId}/${this._rssPath}`;

        return got(url)
            .then(this._parseBlogsRss.bind(this));
    }

    _parseBlogsRss(data) {
        const parser = new Parser();

        return new Promise((resolve, reject) => {
            parser.parseString(data.body, (err, result) => {
                if (err) {
                    return reject(err);
                }

                const items = _.get(result, 'rss.channel[0].item', []);

                resolve(items.map(item => {
                    const linkParts = item.link[0].split('/');

                    return _.assign({ slug: linkParts.pop() }, item);
                }));
            });
        });
    }

    /**
     * Заменяет конструкции в фигурных скобках на
     * @param {String} content
     * @returns {String}
     * @private
     */
    _sanitizeTemplateStrings(content) {
        return content.replace(/{{[^}]*?}}/g, str => {
            const match = str.match(/^{{([^:]*?):/);
            const handlerName = match && match[1];

            if (!Object.hasOwnProperty.call(templateHandlers, handlerName)) {
                return str;
            }

            return templateHandlers[handlerName](str);
        });
    }
}

module.exports = Rss;
