'use strict';

const _ = require('lodash');
const Promise = require('bluebird');
const config = require('yandex-cfg');
const url = require('url');
const { shuffle } = require('shuffle-seed');

const Filterable = require('./abstractions/filterable');

const CardTypesGenerator = require('../lib/card-types-generator');
const { getBunkerNode, getBlogLang } = require('../lib/helper');
const getMapsApi = require('../lib/getMapsApi');
const leftMenu = require('../lib/left-menu');
const { getSeoData } = require('../lib/seo');
const helper = require('../lib/helper');

const SolutionsModel = require('../model/solutions');
const PostModel = require('../model/news/post');

const SECTION = 'contact';
const AGENCIES_PAGE_SIZE = 20;
const MATERIAL_REGEXP = new RegExp('\\.(\\w+(?:\\.tr)?)/adv/(news|solutions/cases)/([\\w-]+)/?$');
const CARD_TYPE_MODES = [{ count: 4, type: 'wide' }, { count: 3 }];

const FILTER_PARAMS = [
    { filterId: 'country' },
    { filterId: 'city', isMultiple: true, hasAllOption: true },
    { filterId: 'certificate', isMultiple: true },
    { filterId: 'budget' },
    { filterId: 'budgetOrder', apiName: 'budget_order' },
    { filterId: 'service', apiName: 'service_slugs' }
];

const NEED_SORT_PRODUCT = ['market'];

function getSeed() {
    return Math
        .random()
        .toString(36)
        .substring(2);
}

function getDefaultAgenciesLists() {
    const { availableGroups } = config.agencies;
    const seed = getSeed();

    return availableGroups.reduce((lists, group) => {
        lists[group] = {
            agencies: [],
            hasMore: false,
            seed
        };

        return lists;
    }, {});
}

class Contacts extends Filterable {
    constructor(req, res, next) {
        super(req, res, next);

        this._filterParams = ['list', 'more', 'permutation'];
    }

    _pickFilters(fromObj, params = {}) {
        const filters = {};
        const { isApi = false } = params;

        for (const filterParam of FILTER_PARAMS) {
            const { filterId, isMultiple = false, hasAllOption = false, apiName } = filterParam;
            let filterValue = fromObj[filterId];

            if (!filterValue) {
                continue;
            }

            if (isMultiple) {
                filterValue = [].concat(filterValue);
            }

            if (isApi && hasAllOption && isMultiple) {
                filterValue = filterValue.filter(option => option !== 'all');
            }

            const filterIdName = isApi && apiName ? apiName : filterId;

            filters[filterIdName] = filterValue;
        }

        return filters;
    }

    async _getCompaniesSlug(dataForFilterApi) {
        const { service_slugs: servicesSlugs } = dataForFilterApi;
        const companiesSlugs = await this._req.backend.companiesSlugs({ ...dataForFilterApi });

        const needFilterByServices = servicesSlugs && servicesSlugs.length;


        if (needFilterByServices) {
            return this._filterByServices(servicesSlugs, companiesSlugs);
        }

        return companiesSlugs;
    }

    _filterByServices(necessaryServices, agenciesSlugs) {
        const agenciesData = _.get(this._req.bunker.agencies2, 'agencies', {});

        const filtredByServicesAgencies = agenciesSlugs.filter( agencySlug => {
            return agenciesData[agencySlug] && agenciesData[agencySlug].products_info;
        }).filter( agencySlug => {
            const productInfo = agenciesData[agencySlug].products_info;
            const agencyServices = productInfo.reduce((itemAgencyServices, productInfoItem) => {
                itemAgencyServices.push(...productInfoItem.service_slugs);

                return itemAgencyServices;
            }, []);

            let result = false;

            if (!Array.isArray(necessaryServices)) {
                return agencyServices.includes(necessaryServices);
            }

            for (let j = 0; j < necessaryServices.length; j = j + 1) {
                result = agencyServices.includes(necessaryServices[j]);

                if (!result) {
                    return result;
                }
            }

            return result;
        });

        return filtredByServicesAgencies;
    }

    async _getCompaniesData(optionsForApi, byWhatProductSort) {
        const agenciesDataFromApi = await this._req.backend.companies(optionsForApi);
        const agenciesDataFromBunker = this._req.bunker.agencies2 &&
            this._req.bunker.agencies2.agencies;

        let agenciesDataFromApiAndBunker = agenciesDataFromApi.map( agencyData => {
            const readyAgencyData = { ...agencyData };

            if (agenciesDataFromBunker && agenciesDataFromBunker[agencyData.slug] ) {
                const productInfo = agenciesDataFromBunker[agencyData.slug].products_info;

                readyAgencyData.productsInfo = productInfo
                    .filter( productsInfo => productsInfo.active);
            }

            return readyAgencyData;
        });

        if (byWhatProductSort) {
            agenciesDataFromApiAndBunker = agenciesDataFromApiAndBunker.sort((current, next) => {
                const metricsCurrent = this._getTotalMetric(current, byWhatProductSort);
                const metricsNext = this._getTotalMetric(next, byWhatProductSort);

                return metricsNext - metricsCurrent;
            });
        }

        return agenciesDataFromApiAndBunker;
    }

    _getTotalMetric(agencyData, byWhatProductSort) {
        const productInfo = agencyData.productsInfo ? agencyData.productsInfo
            .find(productInfoItem => productInfoItem.product_slug === byWhatProductSort) : 0;

        return productInfo ? productInfo.metrics
            .find(metricItem => metricItem.slug === productInfo.total_slug).value : 0;
    }

    _getByWhatProductTotalMetricSort(filters) {
        return filters.certificate.length === 1 &&
        NEED_SORT_PRODUCT.includes(filters.certificate[0]) &&
        filters.certificate[0];
    }

    agencies() {
        const {
            title,
            filtersTemplate,
            description,
            defaultFilters: rawDefaultFilters = {},
            budget: budgetConfig = {}
        } = getBunkerNode(
            this._req.tld,
            this._req.bunker.sources,
            { path: ['contact', 'agencies'] }
        );

        const defaultFilters = this._pickFilters({ ...rawDefaultFilters });

        const filtersForApi = this._pickFilters({
            ...defaultFilters,
            ...this.query
        }, { isApi: true });

        const filtersForFront = this._pickFilters({
            ...defaultFilters,
            ...this.query
        });
        const byWhatProductSort = this._getByWhatProductTotalMetricSort(filtersForFront);
        const metrikaCounters = this._getMetrikaCounters();
        const seo = getSeoData(this._req, {
            section: SECTION,
            titleData: title,
            descriptionData: description
        });
        const regions = getBunkerNode(
            this._req.tld,
            this._req.bunker.sources,
            { path: ['contact', 'agencies', 'regions'] }
        );

        const mapsApi = this._getYMapsUrl();
        const otherGroup = this._getOtherGroup(filtersForApi.country);

        const mainSlugsPromise = this._getCompaniesSlug({
            ...filtersForApi,
            group: 'main'
        });
        const otherSlugsPromise = this._getCompaniesSlug({
            ...filtersForApi,
            group: otherGroup
        });
        const rawFiltersPromise = this._getCompaniesFilters();

        Promise
            .props({
                mainSlugs: mainSlugsPromise,
                otherSlugs: otherSlugsPromise,
                rawFilters: rawFiltersPromise
            })
            .then(({ mainSlugs, otherSlugs, rawFilters }) => {
                const mainSeed = getSeed();
                const otherSeed = getSeed();
                const { 'budget_order': budgetOrder } = filtersForApi;

                return Promise
                    .props({
                        mainAgencies: this._getCompanies({ slugs: mainSlugs, options: {
                            seed: mainSeed,
                            size: AGENCIES_PAGE_SIZE
                        }, budgetOrder, byWhatProductSort } ),
                        otherAgencies: this._getCompanies({ slugs: otherSlugs, options: {
                            seed: otherSeed,
                            size: AGENCIES_PAGE_SIZE
                        }, budgetOrder, byWhatProductSort })
                    })
                    .then(({ mainAgencies, otherAgencies }) => {
                        const agenciesPage = {
                            title,
                            filtersTemplate,
                            description,
                            defaultFilters,
                            budgetConfig,
                            filterUrl: this._buildAjaxPath('companies')
                        };

                        this._res.reactRender({
                            data: {
                                agenciesPage,
                                agencyPage: this._getAgencyPageCommonData(),
                                metrikaCounters,
                                regions,
                                seo,
                                mapsApi
                            },
                            state: {
                                agencies: {
                                    filters: filtersForFront,
                                    rawFilters,
                                    lists: {
                                        ...getDefaultAgenciesLists(),
                                        main: {
                                            agencies: mainAgencies,
                                            seed: mainSeed,
                                            hasMore: mainSlugs.length > AGENCIES_PAGE_SIZE
                                        },
                                        [otherGroup]: {
                                            agencies: otherAgencies,
                                            seed: otherSeed,
                                            hasMore: otherSlugs.length > AGENCIES_PAGE_SIZE
                                        }
                                    }
                                }
                            }
                        });
                    });
            })
            .catch(this._renderError.bind(this));
    }

    _getOtherGroup(countryGeoId) {
        const { groupsByCountryGeoId } = config.agencies;

        return groupsByCountryGeoId[countryGeoId] || groupsByCountryGeoId.default;
    }

    _getCompanies({ slugs, options, budgetOrder, byWhatProductSort }) {
        if (budgetOrder) {
            return this._getOrderedCompanies(
                { slugs, options, budgetOrder, byWhatProductSort: false }
            );
        }
        if (byWhatProductSort) {
            return this._getOrderedCompanies({ slugs, options, budgetOrder, byWhatProductSort });
        }

        return this._getShuffledCompanies(slugs, options);
    }

    _getOrderedCompanies({ slugs, options, budgetOrder, byWhatProductSort = false }) {
        const { size, offset = 0 } = options;
        const slugsChunk = slugs.slice(offset, offset + size);

        return _.isEmpty(slugsChunk) ?
            Promise.resolve([]) :
            this._getCompaniesData({
                slug: slugsChunk,
                'budget_order': budgetOrder
            }, byWhatProductSort);
    }

    _getShuffledCompanies(slugs, { seed, size, offset = 0 }) {
        const slugsChunk = shuffle(slugs, seed).slice(offset, offset + size);

        if (_.isEmpty(slugsChunk)) {
            return Promise.resolve([]);
        }

        return this._getCompaniesData({ slug: slugsChunk })
            .then(agencies => {
                return slugsChunk
                    .map(slug => agencies.find(agency => agency.slug === slug))
                    .filter(Boolean);
            });
    }

    agenciesRedirect() {
        this._res.redirect(301, this._buildPath('contact', 'agencies'));
    }

    agencyRedirect() {
        const { url: site } = _.pick(this.query, ['url']);

        if (!site) {
            return this._renderError('Agency site wasn\'t provided');
        }

        this._req.backend
            .companiesSlugs({ site })
            .then(([agencySlug]) => {
                if (!agencySlug) {
                    return this._renderError(`Agency with site "${site}" not found`);
                }

                this._res.redirect(301, this._buildPath('contact', 'agencies', agencySlug));
            })
            .catch(this._renderError.bind(this));
    }

    async _getCompaniesFilters() {
        const rawFilters = await this._req.backend.companiesFilters();
        const services = _.get(this._req.bunker.agencies2, 'services', {});

        let rowService = Object.values(services);

        rowService = rowService.filter(service => service.active).map(service => {
            return {
                code: service.slug,
                title: service.title.ru,
                certificates: service.product_slugs
            };
        });
        rawFilters.services = rowService;

        return rawFilters;
    }

    agency() {
        const { slug } = this._req.params;
        const metrikaCounters = this._getMetrikaCounters();
        const regions = getBunkerNode(
            this._req.tld,
            this._req.bunker.sources,
            { path: ['contact', 'agencies', 'regions'] }
        );
        const mapsApi = this._getYMapsUrl();
        const agencyPromise = this._getAgencyData(slug);
        const rawFiltersPromise = this._getCompaniesFilters();

        Promise
            .props({
                agency: agencyPromise,
                rawFilters: rawFiltersPromise
            })
            .then(({ agency, rawFilters }) => {
                const agencyPage = {
                    agency,
                    ...this._getAgencyPageCommonData()
                };

                const seo = getSeoData(this._req, {
                    section: SECTION,
                    titleData: agency.name,
                    descriptionData: agency.description
                });

                const { offices = [] } = agency;
                const initialOffice = offices.find(office => office.isMain) || offices[0];
                const selectedId = _.get(initialOffice, 'id', null);

                this._res.reactRender({
                    data: {
                        agencyPage,
                        metrikaCounters,
                        regions,
                        seo,
                        mapsApi
                    },
                    state: {
                        agencies: {
                            rawFilters
                        },
                        agency: {
                            form: {},
                            offices: { selectedId }
                        }
                    }
                });
            })
            .catch(this._renderError.bind(this));
    }

    agencyData() {
        const { slug } = this.body;

        this
            ._getAgencyData(slug)
            .then(agency => this._res.json({ agency }));
    }

    editAgency() {
        const { slug, agency } = this.body;

        const data = _.pick(agency, [
            'name',
            'description',
            'directBudget',
            'site',
            'slug',
            'offices'
        ]);

        return this._req.backend
            // Используется нормализованный логин
            .editCompany(slug, data, this._req.blackbox.login)
            .then(() => this._res.sendStatus(201))
            .catch(error => {
                const errorBody = _.get(error, 'originalError.response.body');

                return this._res
                    .status(error.statusCode || 500)
                    .json(errorBody);
            });
    }

    filterAgencies() {
        const { filters, offset, seed, group } = this.body;
        const { budgetOrder } = filters;

        const byWhatProductSort = this._getByWhatProductTotalMetricSort(filters);

        this._getCompaniesSlug({
            ...this._pickFilters(filters, { isApi: true }),
            group
        })
            .then(slugs => {

                return this
                    ._getCompanies({ slugs, options: {
                        seed,
                        size: AGENCIES_PAGE_SIZE,
                        offset
                    }, budgetOrder,
                    byWhatProductSort } )
                    .then(agencies => {
                        this._res.json({
                            agencies,
                            hasMore: slugs.length > offset + AGENCIES_PAGE_SIZE
                        });
                    });
            });
    }

    offices() {
        const page = 'offices';
        const offices = this._getOffices();

        if (!offices.offices.length) {
            return this._renderError();
        }

        const seoOpts = {
            section: SECTION,
            titleData: _.get(offices, 'title'),
            descriptionData: _.get(offices, 'description')
        };

        const data = {
            section: SECTION,
            page,
            menu: this._getContactsMenu(page),
            offices,
            country: this.query.country,
            contactPageUrl: this._buildPath('contact'),
            mapsApi: getMapsApi(this._req.tld, offices.offices.map(office => {
                return office.map;
            })),
            seo: getSeoData(this._req, seoOpts)
        };

        this._res.renderWithLocals('contacts', data);
    }

    support() {
        const page = 'support';
        const support = getBunkerNode(this._req.tld, this._req.bunker.sources, {
            path: [SECTION, page]
        });

        if (!support) {
            return this._renderError();
        }

        const seoOpts = {
            section: SECTION,
            titleData: _.get(support, 'title'),
            descriptionData: _.get(support, 'description')
        };

        const data = {
            section: SECTION,
            page,
            menu: this._getContactsMenu('/', true),
            support,
            contactPageUrl: this._buildPath('contact'),
            seo: getSeoData(this._req, seoOpts)
        };

        this._res.renderWithLocals('contacts', data);
    }

    _getAgencyPageCommonData() {
        const contactPageUrl = this._buildPath('contact');
        const agenciesPageUrl = this._buildPath('contact', 'agencies');
        const agencyUrl = this._buildAjaxPath('company');
        const metrics = this._req.bunker.agencies2 && this._req.bunker.agencies2.metrics;

        return {
            contactPageUrl,
            agenciesPageUrl,
            agencyUrl,
            metrics
        };
    }

    _getYMapsUrl() {
        return url.format({
            host: config.mapsApi,
            query: { lang: config.ymapsLocales[this._req.tld] || 'ru_RU' }
        });
    }

    async _getAgencyData(slug) {
        const [agency] = await this._getCompaniesData({ slug });

        const materials = await this._getMaterials(agency.materials);
        const seo = getSeoData(this._req, {
            section: SECTION,
            titleData: agency.name,
            descriptionData: agency.description
        });

        return Object.assign(agency, {
            materials,
            seo
        });
    }

    async _getMaterials(materialUrls) {
        const requests = materialUrls.map(async materialUrl => {
            const [, tld, type, slug] = materialUrl.match(MATERIAL_REGEXP);

            const material = {
                type,
                url: materialUrl
            };

            if (type === 'solutions/cases') {
                const caseData = await this._getCaseMaterialData(tld, slug);

                if (!caseData) {
                    return null;
                }

                const date = _.get(caseData, 'solution.date', '');

                return {
                    ...material,
                    data: caseData.solution,
                    date: helper.getDate(date)
                };
            }

            if (type === 'news') {
                const postData = await this._getPostMaterialData(tld, slug);

                if (!postData) {
                    return null;
                }

                return {
                    ...material,
                    data: postData,
                    date: postData.publishDate
                };
            }

            return null;
        });

        const materials = await Promise.all(requests);
        const cardTypesGenerator = new CardTypesGenerator(CARD_TYPE_MODES);

        return materials
            .filter(Boolean)
            .sort((material1, material2) => new Date(material2.date) - new Date(material1.date))
            .map(material => {
                return {
                    ...material,
                    cardType: cardTypesGenerator.next()
                };
            });
    }

    _getCaseMaterialData(tld, slug) {
        const solutions = new SolutionsModel(this._req, {
            section: 'cases',
            slug,
            tld
        });

        return solutions
            .fetch()
            .catch(error => {
                this._req.logger.error(error);

                return null;
            });
    }

    _getPostMaterialData(tld, slug) {
        const post = new PostModel(this._req, {
            blogIdentity: config.blogs.news[tld] || config.blogs.news.default,
            postIdentity: slug,
            query: {
                lang: getBlogLang(tld)
            }
        });

        return post
            .fetch()
            .then(data => {
                return _.pick(data.item, [
                    'publishDate',
                    'approvedTitle'
                ]);
            })
            .catch(error => {
                this._req.logger.error(error);

                return null;
            });
    }

    _getMetrikaCounters() {
        const sectionCounters = config.metrikaCounters
            .filter(counter => counter.section === 'agency');

        return this._req.locals.metrikaCounters.concat(sectionCounters);
    }

    _getContactsMenu(slug, isDefault) {
        const contactNode = _.get(this._req, 'bunker.menu.contact', {});
        const menu = getBunkerNode(this._req.tld, contactNode);

        return leftMenu(menu, { slug, root: this._buildPath('contact'), isDefault });
    }

    _getOffices() {
        let offices = getBunkerNode(this._req.tld, this._req.bunker.sources, {
            path: 'contact.offices'
        });

        const pageFields = ['title', 'description', 'countries', 'date'];
        const page = _.pick(offices, pageFields);

        offices = _.omit(offices, pageFields);
        const citiesOrder = [];

        const citiesDict = _(page.countries)
            .map(country => {
                return country.cities.map(city => {
                    citiesOrder.push(city.name);

                    return _.assign({}, city, { country: country.name });
                });
            })
            .flatten()
            .keyBy('name')
            .value();

        page.offices = citiesOrder.reduce((result, city) => {
            const office = offices[city];

            if (office) {
                result.push(_.assign(office, { city: citiesDict[city] }));
            }

            return result;
        }, []);

        return page;
    }
}

module.exports = Contacts;
