'use strict';

const _ = require('lodash');
const Promise = require('bluebird');
const config = require('yandex-cfg');

const news = require('./news');
const helper = require('../lib/helper');
const seo = require('../lib/seo');

const StatisticsBase = require('./abstractions/statistics');
const Courses = require('../model/edu/online/courses');

const attributesNames = ['service', 'course', 'lesson'];

class OnlineCourses extends StatisticsBase {
    get model() {
        const attributes = _.pick(this._req.params, attributesNames);

        return new Courses(this._req, attributes);
    }

    get page() {
        return this._req.params.lesson ? 'lesson' : 'course';
    }

    course() {
        if (config.regionsWithCourses.indexOf(this._req.tld) < 0) {
            this._req.params.page = this._req.params.service;
            this._req.params.post = this._req.params.course;

            return news.eduPost(this._req, this._res);
        }

        Promise.resolve(this.model.fetch())
            .then(this._addLevels.bind(this))
            .then(this._addMetaData.bind(this))
            .then(this._addMetrikaCounters.bind(this))
            .then(this._render.bind(this))
            .catch(this._renderError.bind(this));
    }

    _addLevels(data) {
        const levelsNode = _.get(data, 'storage._main', {});

        return _.assign({
            levels: helper.getPageLevels(levelsNode || {}),
            shareLevel: helper.getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'share-level'
                })
        }, data);
    }

    _addMetaData(data) {
        const defaultOg = helper.getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'og'
        });
        const title = _.get(data, 'storage.og.title', defaultOg.title);
        const description = _.get(data, 'storage.og.description', defaultOg.description);
        const image = _.get(data, 'storage.og.image', defaultOg.image);
        const seoOpts = {
            section: 'online-courses',
            titleData: title,
            descriptionData: description
        };

        return _.assign({
            og: { title, description, image },
            seo: seo.getSeoData(this._req, seoOpts)
        }, data);
    }

    /* eslint-disable complexity */
    _render(data) {
        const storage = _.get(data, 'storage');

        if (storage) {
            return this._res.renderWithLocals(this.page, data);
        }

        const { service, course, lesson } = this._req.params;
        const coursePath = lesson && [service, course];
        const schoolPath = course && ['online', service];
        const redirectPath = coursePath || schoolPath || 'online';

        this._res.redirect(301, this._buildPath('edu', ...redirectPath));
    }
    /* eslint-enable complexity */
}

module.exports = OnlineCourses;
