'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');
const { getBunkerNode, replaceShy, getPageLevels } = require('../lib/helper');
const { getSeoData, getOgData } = require('../lib/seo');
const Promise = require('bluebird');
const getMapsApi = require('../lib/getMapsApi');

const Base = require('./abstractions/base');
const EventModel = require('../model/edu/event');
const Events = require('../model/edu/events');
const Experts = require('../model/edu/experts');
const ExpertModel = require('../model/edu/expert');
const OnlineServices = require('../model/edu/onlineServices');

const SECTION = 'edu';

function addExpertSeoData(data) {
    const expert = _.get(data, 'expertData.expert', {});
    const og = {
        title: expert.name && (`${data.yandexEduExpert} ${replaceShy(expert.name)}`),
        description: expert.post,
        image: expert.bigPhoto
    };

    return _.assign({
        og,
        seo: {
            title: og.title,
            description: og.description,
            keywords: ''
        }
    }, data);
}

class Edu extends Base {
    index() {
        const page = 'edu';
        const events = new Events(this._req);
        const onlineServices = new OnlineServices(this._req);

        return Promise
            .props({
                page,
                section: SECTION,
                seoSection: 'edu',
                eventsData: events.fetch(),
                services: onlineServices.fetch(),
                eduPageUrl: 'edu'
            })
            .then(this._addRegions.bind(this))
            .then(this._addLevels.bind(this, true))
            .then(this._addSeoData.bind(this))
            .then(this._renderIndex.bind(this))
            .catch(this._renderError.bind(this));
    }

    events() {
        const page = 'events';
        const { query } = this._req;

        const attributes = {
            cities: query.cities || [],
            themes: query.themes || [],
            months: query.months || []
        };
        const events = new Events(this._req, attributes);

        return Promise
            .props({
                section: SECTION,
                page,
                seoSection: 'edu-events',
                eventsData: events.fetch(),
                eventsFilters: attributes,
                eventsPage: events.fetchPage(),
                searchUrl: this._buildPath('edu', 'events')
            })
            .then(this._addEventRegions.bind(this))
            .then(this._addLevels.bind(this, false))
            .then(this._addSeoData.bind(this))
            .then(this._renderList.bind(this, 'eventsData.events', 'eventsPage'))
            .catch(this._renderError.bind(this));
    }

    event() {
        const page = 'event';
        const eventSlug = this._req.params.event;

        const attributes = { eventSlug };
        const event = new EventModel(this._req, attributes);
        const events = new Events(this._req);

        return Promise
            .props({
                section: SECTION,
                page,
                eventData: event.fetch(),
                eventsPage: events.fetchPage()
            })
            .then(this._addEventRegions.bind(this))
            .then(this._addMapsApi.bind(this))
            .then(this._addLevels.bind(this, false))
            .then(this._addEventSeoData.bind(this))
            .then(this._renderEventPage.bind(this))
            .catch(this._renderError.bind(this));
    }

    experts() {
        const page = 'experts';
        const { query } = this._req;

        const attributes = {
            cities: query.cities || [],
            specializations: query.specializations || [],
            workFormats: query.workFormats || []
        };
        const experts = new Experts(this._req, attributes);

        return Promise
            .props({
                section: SECTION,
                page,
                seoSection: 'edu-experts',
                expertsData: experts.fetch(),
                expertsFilters: attributes,
                expertsPage: experts.fetchPage(),
                searchUrl: this._buildPath('edu', 'experts'),
                scrollToFilters: this._req.query.filters
            })
            .then(this._addExpertRegions.bind(this))
            .then(this._addLevels.bind(this, false))
            .then(this._addSeoData.bind(this))
            .then(this._renderList.bind(this, 'expertsData.experts', 'expertsPage'))
            .catch(this._renderError.bind(this));
    }

    expert() {
        const page = 'expert';
        const expertSlug = this._req.params.expert;

        const attributes = { expertSlug };
        const expert = new ExpertModel(this._req, attributes);
        const experts = new Experts(this._req);

        return Promise
            .props({
                section: SECTION,
                page,
                expertData: expert.fetch(),
                expertsPage: experts.fetchPage(),
                searchUrl: this._buildPath('edu', 'experts'),
                eduExpertSettings: getBunkerNode(this._req.tld,
                    this._req.bunker.settings,
                    {
                        path: 'EduExpertSettings'
                    })
            })
            .then(this._addEduExpertInfo.bind(this))
            .then(this._addExpertRegions.bind(this))
            .then(this._addLevels.bind(this, false))
            .then(addExpertSeoData)
            .then(this._renderExpertPage.bind(this))
            .catch(this._renderError.bind(this));
    }

    _addEduExpertInfo(data) {
        const { YandexEduExpert, EduExpert, CertTextGender } = data.eduExpertSettings;

        const gender = data.expertData.expert.gender || 'male';
        const certText = CertTextGender[gender];

        return _.assign(data, {
            yandexEduExpert: YandexEduExpert,
            eduExpert: EduExpert,
            certByYandex: certText
        });
    }

    _addLevels(isMainPage, data) {
        const edu = getBunkerNode(this._req.tld, this._req.bunker.sources, {
            path: 'edu._main'
        });

        return _.assign({
            levels: getPageLevels(edu, level => (isMainPage || level.isEduLevel)),
            shareLevel: getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'share-level'
                }),
            share: config.share
        }, data);
    }

    _addSeoData(data) {
        const seoOpts = {
            section: data.seoSection || 'edu',
            titleData: '',
            descriptionData: ''
        };

        return _.assign({
            seo: getSeoData(this._req, seoOpts),
            og: getOgData(this._req, seoOpts)
        }, data);
    }

    _addEventSeoData(data) {
        const og = _.get(data, 'eventData.event.og', getOgData(this._req, { section: 'edu' }));

        return _.assign({
            og,
            seo: {
                title: og.title,
                description: og.description,
                keywords: ''
            }
        }, data);
    }

    _addMapsApi(data) {
        if (!data.eventData) {
            return data;
        }

        const maps = [data.eventData.event.map];

        return _.assign({
            mapsApi: getMapsApi(this._req.tld, maps)
        }, data);
    }

    _addRegions(data) {
        const edu = getBunkerNode(this._req.tld, this._req.bunker.sources, {
            path: 'edu'
        });

        return edu && _.assign({
            regions: edu.regions || {}
        }, data);
    }

    _addEventRegions(data) {
        return _.assign({
            regions: _.get(data, 'eventsPage.regions', { })
        }, data);
    }

    _addExpertRegions(data) {
        return _.assign({
            regions: _.get(data, 'expertsPage.regions', { })
        }, data);
    }

    _renderIndex(data) {
        const page = _.get(data, 'page');
        const levels = _.get(data, 'levels', []);

        if (!page || _.isEmpty(levels)) {
            return this._renderError();
        }

        this._res.renderWithLocals(page, data);
    }

    _renderList(dataField, pageField, data) {
        const list = _.get(data, dataField);
        const page = _.get(data, pageField, {});

        if (Array.isArray(list) && !_.isEmpty(page)) {
            return this._res.renderWithLocals(data.page, data);
        }

        this._renderError();
    }

    _renderEventPage(data) {
        if (data.eventData) {
            this._res.renderWithLocals(data.page, data);
        } else {
            this._res.redirect(this._buildPath('edu', 'events'));
        }
    }

    _renderExpertPage(data) {
        if (data.expertData) {
            this._res.renderWithLocals(data.page, data);
        } else {
            this._res.redirect(this._buildPath('edu', 'experts'));
        }
    }
}

module.exports = Edu;
