'use strict';

const _ = require('lodash');
const url = require('url');
const Promise = require('bluebird');

const helper = require('../lib/helper');
const sendMail = require('../lib/send-mail');
const { buildPath } = require('../router/path');

function setFields(fields, letter, section) {
    /* eslint-disable complexity */
    fields.forEach(field => {
        if (field.isSelect) {
            // Если в каком-то селекте выбран вариант со списком email-адресов,
            // то подставляем этот список в графу получателей вместо ранее обозначенного.
            const select = _.find(section.fields, { label: field.label });

            if (!select) {
                return;
            }

            const option = _.find(select.options, { value: field.value });

            if (option && !_.isEmpty(option.emails)) {
                letter.to = option.emails;
            }
        } else if (field.isEmail) {
            letter.from = field.value;
        }
    });
    /* eslint-enable complexity */

    return letter;
}

/**
 * Отправка письма со страницы "Разместить рекламу / С помощью менеджера"
 * @param {Object} fields,
 * @param {Object} req
 * @returns {Object}
 */

function getLetterByOrder(fields, req) {
    const { orderId } = fields;
    const forms = helper.getBunkerNode(req.tld, req.bunker.forms);
    let letter = {
        headers: {}
    };

    const page = helper.getBunkerNode(req.tld, req.bunker.sources, {
        path: ['order-pages', orderId.page]
    });
    const section = _.find(page, { id: orderId.section });
    const clientUrl = url.format({
        protocol: 'https',
        hostname: req.headers.host,
        pathname: buildPath(req.locals.url, 'order', orderId.page, orderId.section)
    });

    const subject = section.subject || forms['order-manager'].subject;

    letter.subject = `${subject}: ${clientUrl}`;
    letter.to = section.emails; // сначала получателями выставляем тех, кто указан в бункере
    letter.headers['x-otrs-fromfeedback'] = clientUrl;

    letter = setFields(fields.form, letter, section);

    return letter;
}

async function getAgencyEmailAndCountry(req, slug, officeId) {
    const [agency] = await req.backend.companies({ slug });
    const { offices } = agency;

    const firstOfficeWithFilledEmail = offices.find(office => Boolean(office.email));
    const firstFilledEmail = _.get(firstOfficeWithFilledEmail, 'email', null);
    const selectedOffice = agency.offices.find(office => office.id === officeId);
    const selectedOfficeEmail = _.get(selectedOffice, 'email', null);

    return {
        email: selectedOfficeEmail || firstFilledEmail,
        countryGeoId: _.get(selectedOffice, 'city.country.geoId', null)
    };
}

/**
 * Формирование письма агентству со страницы агентства
 * @param {Object} fields,
 * @param {Object} req
 * @returns {Object}
 */
async function getLetterByAgency(fields, req) {
    const { agency: { slug, officeId } } = fields;
    const forms = helper.getBunkerNode(req.tld, req.bunker.forms);
    const { email, countryGeoId } = await getAgencyEmailAndCountry(req, slug, officeId);
    const letter = {};

    if (!email) {
        return;
    }

    // адреса, которым придет дубликат письма
    const watchers = helper.watchersByAgencyCountry(req, countryGeoId);

    letter.to = _.union([email], watchers);
    letter.subject = forms.settings.agencySubject; // тема этого письма лежит в /forms/tld/settings

    return letter;
}

function renderContactFormError(req, res, error) {
    req.logger.error(error, 'Contact form message not sent');

    return res
        .status(503)
        .send({
            internalCode: '503_MNS',
            message: 'Message not sent'
        });
}

/**
 * Отправка типовых писем из простых форм (со страниц агентств, CTA и т.д.)
 * @param {Object} req,
 * @param {Object} res
 */
exports.contactForm = function (req, res) {
    const fields = _.get(req, 'formData.fields', {});
    const files = _.get(req, 'formData.files');

    Promise.resolve()
        .then(() => {
            if (fields.orderId) {
                return getLetterByOrder(fields, req);
            } else if (fields.agency) {
                return getLetterByAgency(fields, req);
            }
        })
        .then(letter => {
            if (_.isEmpty(letter)) {
                throw new Error('incorrect contact form type');
            }

            letter.fields = fields.form;
            letter.to = helper.getRecipient(req, letter.to);
            letter.files = files;

            return letter;
        })
        .then(letter => sendMail(letter, req))
        .then(() => res.sendStatus(200))
        .catch(error => renderContactFormError(req, res, error));
};

/**
 * Отправка письма из формы "Стать партнером" (/adv/partners/request)
 * @param {Object} req,
 * @param {Object} res
 */
exports.partnersForm = function (req, res) {
    const forms = helper.getBunkerNode(req.tld, req.bunker.forms);
    const { body } = req;

    // Тема и адресаты этого письма лежат в /forms/tld/partners-form
    const form = forms['partners-form'];
    let emails = [];

    if (body.choosenService === 'direct') {
        emails = form.emails.direct[body.goal];
    } else if (body.choosenService === 'directAndMarket') {
        emails = form.emails.direct[body.goal].concat(form.emails.market);
    } else {
        emails = form.emails[body.choosenService];
    }

    const recipients = helper.getRecipient(req, emails);
    const fields = body.step1.concat(body.step2);
    const { subject } = form;

    const sendHandlers = recipients.map(recipient => sendMail({
        to: recipient,
        subject,
        fields
    }, req));

    Promise.all(sendHandlers)
        .then(() => {
            req.logger.info(
                { choosenService: body.choosenService, goal: body.goal },
                'PartnersForm sent'
            );

            res.sendStatus(204);
        })
        .catch(error => {
            req.logger.error(error, 'PartnersForm sending failed!');

            res
                .status(503)
                .send({
                    internalCode: '503_MNS',
                    message: 'Message not sent'
                });
        });
};
