'use strict';

const _ = require('lodash');

const StatisticsBase = require('./abstractions/statistics');
const FreelancersModel = require('../model/freelancers');
const { getSeoData } = require('../lib/seo');
const sendMail = require('../lib/send-mail');
const {
    getBunkerNode,
    getPageLevels,
    getRecipient,
    replaceShy
} = require('../lib/helper');

class Freelancers extends StatisticsBase {
    get pageName() {
        return this._req.params.slug ? 'freelancer' : 'freelancers';
    }

    list() {
        const attributes = _.omit(this.query, ['filters']);
        const freelancersModel = new FreelancersModel(this._req, attributes);

        return freelancersModel.fetchList()
            .bind(this)
            .then(this._addLevels)
            .then(this._addMetaData)
            .then(this._addMetrikaCounters)
            .then(this._renderList)
            .catch(this._renderError);
    }

    freelancer() {
        const { slug } = this._req.params;
        const freelancersModel = new FreelancersModel(this._req, { slug });

        return freelancersModel.fetch()
            .bind(this)
            .then(this._addLevels)
            .then(this._addMetaData)
            .then(this._addMetrikaCounters)
            .then(this._renderFreelancer)
            .catch(this._renderError);
    }

    contact() {
        const formData = _.get(this._req, 'formData');
        const letter = this._buildLetter(formData);

        sendMail(letter, this._req)
            .then(() => this._sendOk())
            .catch(error => {
                this._req.logger.error(error, 'Message not sent');

                return this._res
                    .status(503)
                    .send({
                        internalCode: '503_MNS',
                        message: 'Message not sent'
                    });
            });
    }

    _addLevels(data) {
        const shareLevel = getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'share-level'
        });
        const levelsNode = getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: 'freelancers._main'
            });

        return _.assign({
            levels: getPageLevels(levelsNode),
            shareLevel
        }, data);
    }

    _addMetaData(data) {
        const defaultOg = getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'og'
        });
        const name = replaceShy(_.get(data, 'name', ''));
        const seoOpts = {
            section: 'freelancers',
            titleData: name ? `${name} — ` : ''
        };
        const seoData = getSeoData(this._req, seoOpts);

        return _.assign({
            og: {
                title: seoData.title,
                description: seoData.description,
                image: _.get(data, 'image', defaultOg.image)
            },
            seo: seoData,
            searchUrl: this._buildPath('contact', 'freelancers'),
            section: 'freelancers'
        }, data);
    }

    _renderList(data) {
        if (!_.get(data, 'title')) {
            return this._renderError();
        }

        this._render(_.assign({
            scrollToFilters: this._req.query.filters
        }, data));
    }

    _renderFreelancer(data) {
        if (!_.get(data, 'name')) {
            return this._res.redirect(this._buildPath('contact', 'freelancers'));
        }


        const contactFormUrl = getBunkerNode(
            this._req.tld,
            this._req.bunker.settings,
            { path: 'freelancers-form' }
        );

        this._render(_.assign({
            verificationUrl: this._buildPath('expert', 'certificates?certId='),
            contactFormUrl
        }, data));
    }

    _buildLetter(data) {
        const { contactFormSettings } = getBunkerNode(this._req.tld,
            this._req.bunker.sources, {
                path: 'freelancers'
            });
        const fields = _.get(data, 'fields.form', []);
        const recipient = _.get(data, 'fields.params.recipient', '');
        const bcc = _.get(contactFormSettings, 'recipients', []);

        return {
            from: {
                name: _.find(fields, field => field.name === 'name').value,
                address: _.find(fields, field => field.name === 'email').value
            },
            fields: fields.filter(field => field.name === 'message'),
            subject: contactFormSettings.subject,
            to: getRecipient(this._req, recipient),
            bcc
        };
    }
}

module.exports = Freelancers;
