'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');
const Promise = require('bluebird');

const { getBunkerNode, getPageLevels } = require('../lib/helper');
const { getSeoData } = require('../lib/seo');

const Comments = require('../model/news/comments');
const Journals = require('../model/journals');

const BaseController = require('./abstractions/base');

class MainPageController extends BaseController {
    get pageName() {
        return 'index';
    }

    get bunkerNode() {
        return getBunkerNode(this._req.tld, this._req.bunker.mainpage);
    }

    /**
     * Главная страница
     */
    index() {
        const slugs = this._getNewsSlugs();

        const comments = new Comments(this._req, { slugs });
        const journals = new Journals(this._req);

        Promise
            .props({
                commentsData: comments.getCommentsDataByBlogsSlugs(),
                journal: journals.actual
            })
            .bind(this)
            .then(this._addMetaData)
            .then(this._addLevels)
            .then(this._addInfo)
            .then(this._render)
            .catch(this._renderError);
    }

    /**
     * Получает слаги новостей на главной странице
     * @returns {String[]}
     * @private
     */
    _getNewsSlugs() {
        return _(this.bunkerNode)
            .mapValues(node => _.flatten(node.columns) || node.items || [])
            .values()
            .flatten()
            .map('link')
            .compact()
            .filter(item => !_.startsWith(item, 'http'))
            .uniq()
            .value();
    }

    /**
     * Добавляет SEO- и OG-данные
     * @param {Object} data
     * @returns {Object}
     * @private
     */
    _addMetaData(data) {
        const section = this.pageName;
        const seo = getSeoData(this._req, { section });
        const og = getBunkerNode(this._req.tld, this._req.bunker.settings, { path: 'og' });

        return _.assign({ seo, og }, data);
    }

    /**
     * Добавляет этажи на страницу
     * @param {Object} data
     * @returns {Object}
     * @private
     */
    _addLevels(data) {
        const { share } = config;
        const journal = _.get(data, 'journal');
        const levels = getPageLevels(this.bunkerNode) || [];
        const shareLevel = getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'share-level'
        });

        if (journal) {
            journal.order = 0;

            levels.splice(0, 0, journal);
        }

        return _.assign({ share, levels, shareLevel }, data);
    }

    /**
     * Добавляет дополнительную информацию для отображения страницы
     * @param {Object} data
     * @returns {Object}
     * @private
     */
    _addInfo(data) {
        const section = this.pageName;

        return _.assign({
            section,
            page: section
        }, data);
    }
}

module.exports = MainPageController;
