'use strict';

const _ = require('lodash');
const RSS = require('rss');

const { commonMetrikaCounters } = require('yandex-cfg');
const helper = require('../lib/helper');
const Rss = require('./abstractions/rss');
const seo = require('../lib/seo');

const CANONICAL = /<link rel="canonical" href="(https?:\/\/[^"]*)\/.*?\/?">/gi;
const METRIKA_ID = /w\.yacounter[^\s]*?\s=\snew\sYa\.Metrika\({\s*?id:([^,]*?),/gi;
const METRIKA_WATCH = /<img src="https?:\/\/mc\.yandex\.ru\/watch\/([^?]*?)\?ut=noindex"/gi;

class InstantArticles extends Rss {
    rss() {
        this._rssPath = 'instantArticlesRss';
        this._metrikaCounterId = commonMetrikaCounters[0].id;
        this._siteUrl = helper.getUrlByPathname(this._req, this._req.baseUrl);

        return this._getBlogsRss()
            .then(posts => {
                const feedData = this._getFeedData(posts);

                posts.forEach(post => {
                    const content = _.get(post, 'content:encoded[0]', '');

                    feedData.item({
                        title: _.get(post, 'title[0]', ''),
                        description: _.get(post, 'description[0]', ''),
                        url: `${feedData.site_url}/news/${post.slug}`,
                        date: _.get(post, 'pubDate[0]', ''),
                        'custom_elements': [
                            { 'content:encoded': this._replaceContent(content) }
                        ]
                    });
                });

                return feedData.xml({ indent: true });
            })
            .then(this._sendXML.bind(this))
            .catch(this._renderError.bind(this));
    }

    _getFeedData() {
        const seoData = seo.getSeoData(this._req, { section: 'instant-articles' });

        return new RSS({
            title: seoData.title,
            description: seoData.description,
            'feed_url': `${this._siteUrl}/${this._rssPath}`,
            'site_url': this._siteUrl
        });
    }

    /**
     * Подмена ссылки на каноническую версию и счётчика метрики в контенте
     * @param {String} content
     * @returns {String}
     */
    _replaceContent(content) {
        return this._sanitizeTemplateStrings(content)
            .replace(CANONICAL, (str, blogsUrl) => {
                const urlIdx = str.indexOf(blogsUrl);
                const prefix = str.slice(0, urlIdx);
                const suffix = str.slice(urlIdx + blogsUrl.length);

                return `${prefix}${this._siteUrl}/news${suffix}`;
            })
            .replace(METRIKA_ID, this._replaceMetrikaId.bind(this))
            .replace(METRIKA_WATCH, this._replaceMetrikaId.bind(this));
    }

    /**
     * Подмена счётчика метрики
     * @param {String} str
     * @param {String} metrikaId
     * @returns {String}
     */
    _replaceMetrikaId(str, metrikaId) {
        return str.replace(new RegExp(metrikaId, 'g'), this._metrikaCounterId);
    }
}

module.exports = InstantArticles;
