const _ = require('lodash');
const { getBunkerNode, getPageLevels } = require('../lib/helper');

const Filterable = require('./abstractions/filterable');
const JournalsModel = require('../model/journals');

class Journals extends Filterable {
    constructor(req, res, next) {
        super(req, res, next);

        this._filterParams = ['list', 'lastIdx', 'more'];
    }

    get pageName() {
        return 'journals';
    }

    list() {
        const attributes = _.pick(this.query, ['date', 'theme']);
        const journals = new JournalsModel(this._req, attributes);

        return journals.fetch()
            .bind(this)
            .then(this._addMetaData)
            .then(this._addLevels)
            .then(this._renderJournals)
            .catch(this._renderError);
    }

    filter() {
        const attributes = _.pick(this._req.body, ['startIdx', 'date', 'theme']);
        const journals = new JournalsModel(this._req, attributes);

        return journals.fetch()
            .bind(this)
            .then(this._filterList)
            .catch(this._filterError);
    }

    _addMetaData(data) {
        const metaData = _.get(data, 'metaData', {});

        metaData.og = _.merge(metaData.og, this._getJournalOgData(data));

        return _.assign({}, metaData, data);
    }

    _addLevels(data) {
        const levelsNode = _.get(data, 'levelsData');
        const shareLevel = getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'share-level'
        });

        return _.assign({
            levels: getPageLevels(levelsNode) || [],
            shareLevel
        }, data);
    }

    _renderJournals(data) {
        const metaData = _.get(data, 'metaData');

        if (!metaData) {
            return this._renderError(data);
        }

        const page = this.pageName;
        const searchUrl = this._buildPath('journals');
        const filterUrl = this._buildAjaxPath('journals-filter');

        return this._render(_.assign({ searchUrl, filterUrl, page }, data));
    }

    /**
     * Возвращает OG-данные для журнала
     * @param {Object} data
     * @returns {{ title: String, description: String, image: String } | {}}
     * @private
     */
    _getJournalOgData(data) {
        const { journal } = this._req.query;

        if (!journal) {
            return {};
        }

        const journalData = _.find(data.list, { anchor: journal });

        if (!journalData) {
            return {};
        }

        return _.omitBy(journalData.og, _.isEmpty) || {};
    }
}

module.exports = Journals;
