const _ = require('lodash');
const config = require('yandex-cfg');
const Promise = require('bluebird');

const { getBunkerNode, getBlogLang } = require('../lib/helper');
const { getSeoData, getOgData } = require('../lib/seo');
const timezone = require('../lib/timezone');
const { buildPath } = require('../router/path');

const Filterable = require('./abstractions/filterable');
const Posts = require('../model/news/posts');
const Comments = require('../model/news/comments');
const Categories = require('../model/news/categories');
const Solutions = require('../model/solutions');

class Materials extends Filterable {
    constructor(req, res, next) {
        super(req, res, next);

        this._filterParams = ['list', 'more'];
        this._pageData = this._getPageData();

        if (this._pageData.filters) {
            this._tagsFilters = this._pageData.filters.map(filter => filter.slug);
        }
    }

    listPage() {
        if (!this._pageData.category) {
            // FIXME временный костыль, пока не опубликована английская страница
            return this._res.sendStatus(404);
        }

        this._list(this.query)
            .then(data => {
                const searchUrl = this._buildPath('materials');
                const filterUrl = this._buildAjaxPath('materials-filter');
                const { categoryId } = this.query;

                data = _.assign({ searchUrl, filterUrl, categoryId }, data);

                this._res.renderWithLocals('materials-list', data);
            })
            .catch(error => {
                this._req.logger.error(error);

                this._res.sendStatus(404);
            });
    }

    filter() {
        return this._list(this._req.body)
            .bind(this)
            .then(({ lastPostId, lastCaseSlug, posts, more }) =>
                this._res.json({ lastPostId, lastCaseSlug, posts, more }))
            .catch(error => {
                this._req.logger.error(error);

                this._res.sendStatus(404);
            });
    }

    _getPostsAttributes(queryParams) {
        const unionParams = _.union([
            'lastPostId',
            'lastCaseSlug',
            'categoryId'
        ], this._tagsFilters);
        const query = _.pick(queryParams, unionParams);

        query.from = query.lastPostId;
        query.size = config.limitMaterials;
        query.lang = getBlogLang(this._req.tld);
        query.categoryId = query.categoryId || this._pageData.category;

        const filteredTags = _.pick(query, this._tagsFilters);
        const tags = _.map(filteredTags,
            tagsBlock => _.isArray(tagsBlock) ? tagsBlock.join(',') : tagsBlock
        );

        if (!_.isEmpty(tags)) {
            query.tags = tags;
        }

        return {
            blogIdentity: config.blogs.news[this._req.tld] || config.blogs.news.default,
            query,
            type: 'news'
        };
    }

    _getCategoryInfo(post, categories) {
        const categoriesObject = _.keyBy(categories, category => category._id);

        const priorityCategories = getBunkerNode(
            this._req.tld,
            this._req.bunker.settings,
            { path: 'materials.firstCategories' }

        ) || [];

        if (!_.isEmpty(post.categoryIds)) {
            const compareCategories = (category1, category2) => {
                const is1Priority = priorityCategories.includes(category1.displayName);
                const is2Priority = priorityCategories.includes(category2.displayName);

                // Если обе категории приоритетные или нет – сортируем в алфавитном порядке
                if ((is1Priority && is2Priority) || !(is1Priority || is2Priority)) {
                    return category1.displayName.localeCompare(category2.displayName);
                }

                if (is1Priority) {
                    return -1;
                }

                if (is2Priority) {
                    return 1;
                }
            };

            post.categories = post.categoryIds
                .map(categoryId => categoriesObject[categoryId])
                .sort(compareCategories);
        }
    }

    _list(params) {
        const attributes = this._getPostsAttributes(params);
        const posts = new Posts(this._req, attributes);
        const categories = new Categories(this._req, attributes);
        const solutions = new Solutions(this._req, {
            ...attributes,
            type: 'solutions'
        });


        return Promise
            .props({
                posts: posts.fetch(),
                categories: categories.fetch(),
                cases: solutions.fetchCases('cases2')
            })
            .then(Posts.unwrap)
            .then(this._addCommentsCountByPost.bind(this))
            .then(obj => this._addBlogUrl(this._req.locals.url, obj))
            .then(obj => {
                this._setCheckedFilters(this._pageData.filters, this.query);
                const data = _.assign({}, this._pageData, obj);

                if (data.posts) {
                    data.posts.forEach(post => {
                        post.date = timezone.postDate(post.publishDate,
                            _.get(data, 'blog.localeTimeZone'));
                        post.url = buildPath(this._req.locals.url, 'news', post.slug);
                        post.commentsCount = data.commentsCounts[post._id];
                    });
                }

                if (data.posts && data.categories) {
                    data.posts.forEach(post => {
                        this._getCategoryInfo(post, data.categories);
                    });
                }

                return data;
            })
            .then(data => {
                const unionPosts = data.posts
                    .concat(data.cases)
                    .sort((post1, post2) => new Date(post2.date) - new Date(post1.date))
                    .slice(0, attributes.query.size);

                const hasNextPost = _.get(_.last(data.posts), 'hasNext');
                const isPostsSliced = unionPosts.length < data.posts.length + data.cases.length;

                data.more = hasNextPost || isPostsSliced;
                data.posts = unionPosts;

                const lastPost = _.findLast(unionPosts, post => post.publishDate);
                const lastCase = _.findLast(unionPosts, post => !post.publishDate);

                data.lastPostId = lastPost && lastPost._id;
                data.lastCaseSlug = lastCase ? lastCase.slug : null;

                return data;
            });
    }

    _setCheckedFilters(filters, query) {
        filters.forEach(filter => {
            let filterQuery = query[filter.slug];

            if (!_.isArray(filterQuery)) {
                filterQuery = [filterQuery];
            }

            filter.values = filter.values.map(value => {
                value.checked = _.includes(filterQuery, value.slug);

                return value;
            });
        });

        return filters;
    }

    _getPageData() {
        const seoOpts = {
            section: 'materials-list',
            titleData: '',
            descriptionData: ''
        };

        const data = getBunkerNode(
            this._req.tld,
            this._req.bunker.sources,
            { path: 'materials' }
        );

        data.page = 'materials-list';
        data.seo = getSeoData(this._req, seoOpts);
        data.og = getOgData(this._req, seoOpts);

        return data;
    }

    _addBlogUrl(host, obj) {
        if (_.isPlainObject(obj.blog)) {
            obj.blog.blogUrl = buildPath(host, 'news');
        }

        return obj;
    }

    _addCommentsCountByPost(obj) {
        const entityIds = obj.posts.map(post => post._id);
        const comments = new Comments(this._req);

        return comments
            .getCommentsCountByCmntIds(entityIds)
            .then(commentsCounts => {
                obj.commentsCounts = commentsCounts;

                return obj;
            });
    }
}

module.exports = Materials;
