'use strict';

const _ = require('lodash');
const Promise = require('bluebird');
const striptags = require('striptags');
const config = require('yandex-cfg');

const timezone = require('../lib/timezone');
const { buildPath, buildAjaxPath } = require('../router/path');

const Blog = require('../model/news/blog');
const Comments = require('../model/news/comments');
const Journals = require('../model/journals');
const Posts = require('../model/news/posts');
const Post = require('../model/news/post');
const Tags = require('../model/news/tags');
const Tag = require('../model/news/tag');
const RelatedArticles = require('../model/news/relatedArticles');
const Page = require('../model/page');
const seo = require('../lib/seo');
const {
    getBunkerNode,
    getBlogLang,
    getPageLevels,
    getArticleCta,
    replaceHtmlEntities
} = require('../lib/helper');

const _getPostDescription = function (post) {
    if (post.approvedPreview) {
        return replaceHtmlEntities(striptags(post.approvedPreview.html));
    }

    if (post.approvedBody) {
        return replaceHtmlEntities(striptags(post.approvedBody.html));
    }
};

const _addOgProperties = function (data) {
    const { post } = data;
    const postTitle = post.approvedTitle || post.proposedTitle;

    data.og = {
        title: postTitle,
        description: _getPostDescription(post),
        image: _.get(post, 'socialImage.orig.fullPath')
    };

    return data;
};

const _addBlogUrl = function (host, obj) {
    if (_.isPlainObject(obj.blog)) {
        obj.blog.blogUrl = buildPath(host, 'news');
    }

    return obj;
};

const _addCommentsCountByPost = function (req, obj) {
    const entityIds = obj.posts.map(post => post._id);
    const comments = new Comments(req);

    return comments
        .getCommentsCountByCmntIds(entityIds)
        .then(commentsData => {
            obj.commentsCounts = commentsData;

            return obj;
        });
};

const _list = function (req) {
    const query = _.pick(req.query, ['tag', 'year', 'month', 'from']);

    query.size = config.limitNews;
    query.lang = getBlogLang(req.tld);

    let data = {
        page: 'news',
        dateFilterEnabled: query.year && query.month,
        tagFilterEnabled: query.tag
    };
    const attributes = {
        blogIdentity: config.blogs.news[req.tld] || config.blogs.news.default,
        query,
        type: 'news'
    };

    const blog = new Blog(req, attributes);
    const posts = new Posts(req, attributes);
    const tags = new Tags(req, attributes);
    const tag = new Tag(req, attributes);
    const journals = new Journals(req);

    return Promise
        .props({
            blog: blog.fetch(),
            posts: posts.fetch(),
            tags: tags.fetch(),
            tag: tag.fetch(),
            journal: journals.forNews
        })
        .then(Posts.unwrap)
        .then(obj => _addCommentsCountByPost(req, obj))
        .then(obj => _addBlogUrl(req.locals.url, obj))
        .then(obj => {
            data = _.assign(data, obj);

            if (data.posts) {
                data.posts.forEach(post => {
                    post.date = timezone.postDate(post.publishDate,
                        _.get(data, 'blog.localeTimeZone'));
                    post.url = buildPath(req.locals.url, 'news', post.slug);
                    post.commentsCount = data.commentsCounts[post._id];
                });
            }

            return data;
        });
};

const _getNextLesson = function (dataObj) {
    const { sections } = dataObj.gotData;

    const urls = sections.reduce((result, section) => {
        const items = section.data.items || [];

        return result.concat(items.reduce((res, lesson) => {
            res.push(lesson.url);

            return res;
        }, []));
    }, []);

    const reg = new RegExp(`${dataObj.req.params.post}$`);
    const index = _.findIndex(urls, lessonUrl => {
        return lessonUrl.match(reg);
    });

    return urls[index + 1];
};

/**
 * Возвращает OG-данные для новостной страницы с тестом
 * @param {Object} req
 * @param {Object} data
 * @param {String} postSlug
 * @returns {{ title: String, description: String, image: String } | {}}
 * @private
 */
const _getTestOgData = (req, data, postSlug) => {
    const { res } = req.query;
    const result = parseInt(res, 10);

    if (_.isNaN(result)) {
        return {};
    }

    const newsTestsNode = getBunkerNode(req.tld, req.bunker['news-tests']);
    const newsTestData = _(newsTestsNode)
        .values()
        .find(newsTest => newsTest.slug === postSlug);

    const og = _(newsTestData)
        .get('settings', [])
        .find(setting => setting.result === result);

    return _.omitBy(og, _.isEmpty) || {};
};

const _initCmntChat = function (req, data) {
    const {
        approvedTitle: name,
        slug,
        _id: entityId
    } = data.post;

    const initData = {
        name,
        description: '',
        entityId,
        entityUrl: `${config.host}${req.tld}/adv/news/${slug}`
    };

    const comments = new Comments(req, { initData });

    return comments
        .init()
        .then(isPostInited => {
            data.isPostInited = isPostInited;

            return data;
        });
};

// FIXME: официальный ответ
// eslint-disable-next-line
const _getOfficialToken = function (req, data) {
    const login = _.get(req, 'blackbox.raw.login', '');
    const { logins } = config.cmnt.officiality;

    if (!logins.includes(login)) {
        data.officialToken = null;

        return Promise.resolve(data);
    }

    const uid = _.get(req, 'blackbox.uid', '');
    const { _id: entityId } = data.post;

    const signData = {
        uid: Number(uid),
        entities: [
            {
                entityId,
                items: [
                    { type: 'official', isOfficial: true }
                ]
            }
        ]
    };

    const comments = new Comments(req, { signData });

    return comments
        .sign()
        .then(token => {
            data.officialToken = token;

            return data;
        });
};

exports.listPage = function (req, res) {
    _list(req)
        .then(data => {
            const seoOpts = {
                section: 'news-list',
                titleData: '',
                descriptionData: ''
            };

            data.page = 'news-list';
            data.seo = seo.getSeoData(req, seoOpts);

            const newsPage = getBunkerNode(req.tld, req.bunker.sources, { path: 'news' });

            data = _.assign({}, newsPage, data);

            res.renderWithLocals('news-list', data);
        })
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.listJson = function (req, res) {
    _list(req)
        .then(data => res.json(data.posts))
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.post = function (req, res) {
    const mainpage = getBunkerNode(req.tld, req.bunker.mainpage);
    const currentUrl = req.locals.url;

    let data = {
        page: 'news',
        isPlainPost: true,
        levels: getPageLevels(mainpage, level => level.isPromo),
        shareLevel: getBunkerNode(req.tld, req.bunker.settings, {
            path: 'share-level'
        }),
        instantArticlesPageId: config.instantArticlesPageId
    };
    const attributes = {
        blogIdentity: config.blogs.news[req.tld] || config.blogs.news.default,
        postIdentity: req.params.post,
        query: {
            lang: getBlogLang(req.tld)
        }
    };

    const blog = new Blog(req, attributes);
    const post = new Post(req, attributes);

    Promise
        .props({
            blog: blog.fetch(),
            post: post.fetch()
        })
        .then(Post.unwrap)
        .then(obj => _initCmntChat(req, obj))
        // .then(obj => _getOfficialToken(req, obj))
        .then(obj => _addBlogUrl(currentUrl, obj))
        .then(_addOgProperties)
        .then(obj => {
            data = _.assign(data, obj);

            const seoOpts = {
                section: `${data.page}-post`,
                titleData: _.get(data, 'post.approvedTitle'),
                descriptionData: _.get(data, 'post.approvedPreview.source')
            };
            const postSlug = _.get(data.post, 'slug');
            const blogSlug = _.get(data.blog, 'slug');
            const isReadonlyComments = getBunkerNode(req.tld, req.bunker.settings, {
                path: 'isReadonlyComments'
            });

            data.post.date = timezone.postDate(data.post.publishDate,
                _.get(data, 'blog.localeTimeZone'));

            data.post.commentsData = {
                entityId: _.get(data.post, '_id'),
                loader: config.cmnt.loader,
                lang: req.language,
                tld: req.tld,
                apiKey: config.cmnt.apiKey,
                isReadonlyComments,
                shouldScrollToComments: Boolean(req.query.comments),
                commentId: req.query.commentId,
                isPostInited: data.isPostInited,
                officialToken: data.officialToken
            };

            data.meta = { ...data.meta, noindex: _.get(data, 'post.isOutdated') };
            data.seo = seo.getSeoData(req, seoOpts);
            data.og = _.merge(data.og, _getTestOgData(req, data, postSlug));
            data.articleCta = getArticleCta(req, _.get(data, 'post.tags.0'));
            data.post.relatedArticlesUrl = buildAjaxPath(
                currentUrl,
                'news',
                blogSlug,
                postSlug,
                'relatedArticles'
            );

            res.renderWithLocals('news', data);
        })
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.eduPost = function (req, res) {
    const course = req.params.page;
    let data = {
        page: 'news',
        section: 'edu',
        backLink: buildPath(req.locals.url, 'edu', course)
    };
    const blogIdentity = config.blogs.edu[req.tld] || config.blogs.edu.default;

    if (!blogIdentity) {
        res.sendStatus(404);

        return;
    }

    const attributes = {
        blogIdentity,
        postIdentity: req.params.post,
        query: {
            lang: getBlogLang(req.tld)
        }
    };

    const blog = new Blog(req, attributes);
    const post = new Post(req, attributes);
    const page = new Page(req, {
        page: course,
        section: 'edu'
    });

    Promise
        .props({
            blog: blog.fetch(),
            post: post.fetch(),
            nextLesson: page
                .getData()
                .then(gotData => ({ req, gotData }))
                .then(_getNextLesson)
        })
        .then(Post.unwrap)
        .then(obj => {
            data = _.assign(data, obj);

            const seoOpts = {
                section: data.section,
                titleData: _.get(data, 'post.approvedTitle'),
                descriptionData: _.get(data, 'post.approvedPreview.source')
            };

            data.seo = seo.getSeoData(req, seoOpts);
            res.renderWithLocals('news', data);
        })
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.relatedArticles = function (req, res) {
    const attributes = {
        blogIdentity: req.params.blogIdentity,
        postIdentity: req.params.postIdentity
    };
    const relatedArticles = new RelatedArticles(req, attributes);

    return relatedArticles
        .fetch()
        .then(result => res.json(result))
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};
