'use strict';

const _ = require('lodash');
const Promise = require('bluebird');

const StatisticsBase = require('./abstractions/statistics');

const helper = require('../lib/helper');
const seo = require('../lib/seo');
const OnlineService = require('../model/edu/online/service');

class OnlineEdu extends StatisticsBase {
    index() {
        const page = 'online-edu';
        const hasPage = helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: 'edu.online._main'
            });

        return Promise
            .resolve({ page, hasPage })
            .then(this._addMainPageLevels.bind(this))
            .then(this._addMainPageMetaData.bind(this))
            .then(this._addMetrikaCounters.bind(this))
            .then(this._renderIndex.bind(this))
            .catch(this._renderError.bind(this));
    }

    service() {
        const [ , , , service, lesson] = this._req.path.split('/');
        const attributes = { service, lesson };
        const onlineService = new OnlineService(this._req, attributes);

        return Promise
            .props({
                serviceData: onlineService.fetch()
            })
            .then(data => _.assign({ requestData: attributes }, data))
            .then(this._addLevels.bind(this))
            .then(this._addMetaData.bind(this))
            .then(this._addMetrikaCounters.bind(this))
            .then(this._render.bind(this))
            .catch(this._renderError.bind(this));
    }

    _addMainPageLevels(data) {
        const edu = helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: 'edu.online._main'
            });

        return _.assign({
            levels: helper.getPageLevels(edu),
            shareLevel: helper.getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'share-level'
                })
        }, data);
    }

    _addMainPageMetaData(data) {
        const defaultOg = helper.getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'og'
        });
        const mainPage = helper.getBunkerNode(this._req.tld,
            this._req.bunker.sources,
            {
                path: 'edu.online._main'
            });
        const title = _.get(mainPage, 'title', defaultOg.title);
        const description = _.get(mainPage, 'description', defaultOg.description);
        const image = _.get(mainPage, 'image', defaultOg.image);
        const keywords = _.get(mainPage, 'keywords', '');

        return _.assign({
            og: { title, description, image },
            seo: { title, description, keywords },
            regions: mainPage && mainPage.regions
        }, data);
    }

    _addLevels(data) {
        const serviceData = _.get(data, 'serviceData.data', {});
        const levelsData = serviceData.levelsData || {};

        return _.assign({
            levels: helper.getPageLevels(serviceData.levels || {}),
            levelsTitle: levelsData.title,
            levelsText: levelsData.text,
            shareLevel: helper.getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'share-level'
                })
        }, data);
    }

    _addMetaData(data) {
        const defaultOg = helper.getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'og'
        });
        const title = _.get(data, 'serviceData.data.og.title', defaultOg.title);
        const description = _.get(data, 'serviceData.data.og.description', defaultOg.description);
        const image = _.get(data, 'serviceData.data.og.image', defaultOg.image);
        const seoOpts = {
            section: 'online-edu',
            titleData: title,
            descriptionData: description
        };

        return _.assign({
            og: { title, description, image },
            seo: seo.getSeoData(this._req, seoOpts),
            regions: _.get(data, 'serviceData.data.regions', { }),
            section: _.get(data, 'serviceData.section')
        }, data);
    }

    _renderIndex(data) {
        if (data.hasPage) {
            this._res.renderWithLocals(data.page, data);
        } else {
            this._res.redirect(301, this._buildPath('edu'));
        }
    }

    _render(data) {
        if (!data.serviceData) {
            return this._res.redirect(301, this._buildPath('edu', 'online'));
        }

        if (data.requestData.lesson && data.serviceData.page === 'online-edu-service') {
            const slug = _.get(data, 'serviceData.data.slug', '');

            return this._res.redirect(301, this._buildPath('edu', 'online', slug));
        }

        const { page } = data.serviceData;

        return this._res.renderWithLocals(page, _.assign({ page }, data));
    }
}

module.exports = OnlineEdu;
