'use strict';

const _ = require('lodash');
const assert = require('http-assert');
const striptags = require('striptags');

const seo = require('../lib/seo');
const helper = require('../lib/helper');

const Base = require('./abstractions/base');
const PageModel = require('../model/page');

class Page extends Base {
    constructor(req, res, next) {
        super(req, res, next);

        this._isAdProduct = false;
    }

    get pageName() {
        return this._isAdProduct ? 'ad-products' : 'products';
    }

    get section() {
        return _.get(this._req, 'params.section');
    }

    get page() {
        return _.get(this._req, 'params.page', '_main');
    }

    get seoSection() {
        return this.section;
    }

    get Model() {
        return PageModel;
    }

    get() {
        assert(this.section, 404, 'Not found');

        const page = new this.Model(this._req, {
            section: this.section,
            page: this.page
        });

        return page
            .getData()
            .then(this._checkIsAd.bind(this))
            .then(this._extendData.bind(this))
            .then(this._addSeoInfo.bind(this))
            .then(this._render.bind(this))
            .catch(this._renderError.bind(this));
    }

    rss() {
        assert(this.section, 404, 'Not found');

        const page = new this.Model(this._req, {
            section: this.section,
            page: this.page
        });

        return page
            .getRss()
            .then(this._sendXML.bind(this))
            .catch(this._renderError.bind(this));
    }

    _checkIsAd(data) {
        this._isAdProduct = Boolean(data.isAdProduct);

        return data;
    }

    /* eslint-disable complexity */
    _extendData(data) {
        const levels = data.levels && data.levels.length && _(data.levels)
            .map(level => _.get(this._req.bunker, level))
            .compact()
            .values();

        let shareLevel = data['share-level'];

        if ((!shareLevel || !shareLevel.enabled) && data.addShareLevel) {
            shareLevel = helper.getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'share-level'
                });
        }

        return _.assign({
            section: this.section,
            page: data.addShareNode && this.pageName,
            isRoot: this.page === '_main',
            productsSettings: helper.getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'products.text'
                }),
            levels,
            shareLevel: data.addShareLevel && shareLevel
        }, _.pick(data, ['product', 'media', 'sections', 'menu', 'regions', 'og', 'queryParams']));
    }
    /* eslint-enable complexity */

    _addSeoInfo(data) {
        const title = _.get(data, 'product.title', '');
        const text = striptags(_.get(data, 'product.text', ''));
        const ownSeoData = _.get(data, 'product.seo', {});
        const seoOpts = {
            section: this.seoSection,
            titleData: title,
            descriptionData: text
        };
        const defaultOg = { title, description: text };
        const seoData = seo.getSeoData(this._req, seoOpts);

        _.set(data, 'meta.noindex', _.get(data, 'product.noindex', false));

        return _.assign(data, {
            seo: {
                title: ownSeoData.title || seoData.title,
                description: ownSeoData.description || seoData.description,
                keywords: ownSeoData.keywords || seoData.keywords
            },
            og: _.get(data, 'og', defaultOg)
        });
    }

    _renderError(error) {
        if (error.internalCode === '301_PMP') {
            this._res.redirect(301, error.location);
        } else {
            this._req.logger.error(error, 'Render error');

            this._res.sendStatus(404);
        }
    }
}

module.exports = Page;
