'use strict';

const _ = require('lodash');
const Promise = require('bluebird');
const config = require('yandex-cfg');

const timezone = require('../lib/timezone');
const { buildPath } = require('../router/path');
const { getBlogLang, getArticleCta, getBunkerNode } = require('../lib/helper');

const Partners = require('../model/partners');
const Blog = require('../model/news/blog');
const Posts = require('../model/news/posts');
const Post = require('../model/news/post');
const Tags = require('../model/news/tags');
const Tag = require('../model/news/tag');
const seo = require('../lib/seo');

const SECTION = 'partners';

const _news = function (req) {
    const query = _.pick(req.query, ['tag', 'year', 'month', 'from']);

    query.size = config.limitPartnersNews;
    query.lang = getBlogLang(req.tld);

    const page = 'news';
    const attributes = {
        blogIdentity: config.blogs.partners[req.tld] || config.blogs.partners.default,
        page,
        query,
        type: 'partners-news'
    };
    const blog = new Blog(req, attributes);
    const posts = new Posts(req, attributes);
    const tags = new Tags(req, attributes);
    const tag = new Tag(req, attributes);
    const partners = new Partners(req, attributes);

    return Promise
        .props({
            access: req.access,
            blog: blog.fetch(),
            posts: posts.fetch(),
            tags: tags.fetch(),
            tag: tag.fetch(),
            partners: partners.fetch()
        })
        .then(Posts.unwrap)
        .then(obj => {
            let data = {
                tag: query.tag,
                section: SECTION,
                partnersPageUrl: buildPath(req.locals.url, 'partners'),
                page,
                menu: obj.partners.menu,
                regions: obj.partners.regions,
                showNewspageControls: true
            };

            data = _.assign(data, obj);

            if (data.posts) {
                data.posts.forEach(post => {
                    post.date = timezone.postDate(post.publishDate,
                        _.get(data, 'blog.localeTimeZone'));
                    post.url = buildPath(req.locals.url, 'partners', 'news', post.slug);
                });
            }

            return data;
        });
};

exports.newsJson = function (req, res) {
    _news(req)
        .then(data => res.json(data.posts))
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.newsPage = function (req, res) {
    _news(req)
        .then(data => {
            const seoOpts = {
                section: SECTION,
                titleData: _.get(data, 'partners.info.title'),
                descriptionData: _.get(data, 'partners.info.text')
            };

            const newsPageData = getBunkerNode(
                req.tld,
                req.bunker.sources,
                { path: 'partners.news-page' }
            );

            data = _.assign({}, newsPageData, data);
            data.seo = seo.getSeoData(req, seoOpts);
            res.renderWithLocals('partners-news', data);
        })
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.post = function (req, res) {
    const attributes = {
        page: 'news',
        blogIdentity: config.blogs.partners[req.tld] || config.blogs.partners.default,
        postIdentity: req.params.post,
        query: {
            lang: getBlogLang(req.tld)
        }
    };

    const blog = new Blog(req, attributes);
    const post = new Post(req, attributes);
    const partners = new Partners(req, attributes);

    Promise
        .props({
            access: req.access,
            blog: blog.fetch(),
            post: post.fetch(),
            partners: partners.fetch()
        })
        .then(Post.unwrap)
        .then(obj => {
            let data = {
                menu: obj.partners.menu,
                section: SECTION,
                regions: obj.partners.regions
            };

            if (!obj.partners.hasAccess) {
                req.params.pageName = 'enter';

                return exports.partners(req, res);
            }

            data.articleCta = getArticleCta(req, _.get(obj, 'post.tags.0'));

            const postDate = timezone.postDate(_.get(obj, 'post.publishDate'),
                _.get(obj, 'blog.localeTimeZone'));

            data = _(data)
                .assign(obj)
                .set('blog.blogUrl', buildPath(req.locals.url, 'partners'))
                .set('post.date', postDate)
                .value();

            const seoOpts = {
                section: SECTION,
                titleData: _.get(data, 'post.approvedTitle'),
                descriptionData: _.get(data, 'post.approvedPreview.source')
            };

            data.meta = { ...data.meta, noindex: _.get(data, 'post.isOutdated') };
            data.seo = seo.getSeoData(req, seoOpts);
            res.renderWithLocals('partners-news', data);
        })
        .catch(error => {
            req.logger.error(error);

            res.sendStatus(404);
        });
};

exports.partners = function (req, res) {
    const { blockName, pageName } = req.params;
    const attributes = { block: blockName, page: pageName };
    const partners = new Partners(req, attributes);
    let data = {
        section: SECTION,
        partnersPageUrl: buildPath(req.locals.url, 'partners')
    };

    return partners
        .fetch()
        .then(partnersData => {
            if (!partnersData) {
                return res.sendStatus(404);
            }

            if (!partnersData.hasAccess) {
                partnersData.page = 'enter';
            }

            data = _.assign(data, partnersData);

            const seoOpts = {
                section: data.section,
                titleData: _.get(data, 'info.title'),
                descriptionData: _.get(data, 'info.text')
            };

            data.seo = seo.getSeoData(req, seoOpts);
            data.meta = { ...data.meta, noindex: _.get(data, 'info.noindex', false) };
            res.renderWithLocals('partners', data);
        })
        .catch(err => {
            if (err.internalCode === '403_PFU') {
                req.params.pageName = 'enter';

                return exports.partners(req, res);
            }

            if (err.internalCode === '301_PRE') {
                return res.redirect(301, err.redirect);
            }

            res.sendStatus(404);
        });
};

exports.enter = function (req, res) {
    if (req.partnersAccess !== 'allowed') {
        req.params.pageName = 'enter';

        return exports.partners(req, res);
    }
    res.redirect(buildPath(req.locals.url, 'partners'));
};

exports.main = function (req, res) {
    if (req.partnersAccess === 'allowed' && req.tld === 'ru') {
        return exports.newsPage(req, res);
    }

    req.params.pageName = 'cooperation-public';
    exports.partners(req, res);
};
