'use strict';

const _ = require('lodash');
const prices = require('../lib/prices');
const helper = require('../lib/helper');
const { buildPath } = require('../router/path');
const seo = require('../lib/seo');

const SECTION = 'prices';

/**
 * Возвращает все прайс-листы продукта
 * Прайс-листы имеют свойство block: 'product-price'
 * @param {Object} product - нода продукта
 * @param {String} url - URL запроса
 * @returns {Array}
 */
const getPriceNodes = function (product, url) {
    return _(product)
        .values()
        .filter(section => _.isObject(section) &&
            section.enabled && section.block === 'product-price')
        .map(price => prices.buildPrice(product.tables, price, url))
        .value();
};

/**
 * Проходит по всем продуктам и возвращает массив объектов,
 * хранящих название продукта и его прайсы.
 * @param {Object} pricesPage - объект текущей страницы
 * @param {String} type - что хотим получить: tables или prices
 * @param {Object} root - нода /adv-www/sources/lang/
 * @param {Object} pagesRelations - отношения между продуктовыми нодами и их слагами
 * @param {String} url - URL запроса
 * @returns {Array|Object}
 */
// eslint-disable-next-line max-params
const getProductsPrices = function (pricesPage, type, root, pagesRelations, url) {
    const pricesPages = root['prices-pages'];
    const tablesPages = _.values(root['tables-pages']) || [];
    const commonTables = tablesPages.map(table => prices.buildTable(table));
    const sections = pricesPages.roots;
    const sectionsDict = _.keyBy(pagesRelations, 'node');

    /* eslint-disable complexity */
    return _.chain(root)
        .pick(sections)
        .reduce((result, n, key) => {
            const product = _.mapValues(n, obj => _.set(obj, 'sectionName', key));

            return _.assign(result, product);
        }, {})
        .thru(products => pricesPage.products.reduce((result, key) => {
            if (!_.isObject(key)) {
                key = { slug: key };
            }

            const product = products[key.slug];

            if (!product) {
                return result;
            }

            const productSlug = _.get(sectionsDict, [product.sectionName, 'slug']);

            product.tables = product.tables || [];

            const ownTables = type === 'common' ? product.tables
                .map(table => prices.buildTable(table)) : [];

            result.push({
                id: key.slug,
                url: buildPath(url, 'products', productSlug, `${key.slug}#price`),
                title: key.title || product.title,
                abovetitle: key.abovetitle,
                tables: ownTables.concat(commonTables),
                prices: type === 'list' ? getPriceNodes(product) : []
            });

            return result;
        }, []))
        .values()
        .flatten()
        .value();
};

exports.common = function (req, res) {
    const root = helper.getBunkerNode(req.tld, req.bunker.sources);
    const pricesPage = _.get(root, ['prices-pages', 'common']);

    if (!pricesPage) {
        res.sendStatus(404);

        return;
    }

    const data = {
        page: SECTION,
        type: 'common',
        regions: pricesPage.regions
    };

    data.products = {
        title: pricesPage.title,
        text: pricesPage.text,
        data: pricesPage.data.map(block => ({
            footer: block.footer,
            list: getProductsPrices(block, 'common', root, req.pagesRelations, req.locals.url)
        }))
    };

    const seoOpts = {
        section: SECTION,
        titleData: _.get(data, 'products.title'),
        descriptionData: _.get(data, 'products.text')
    };

    data.seo = seo.getSeoData(req, seoOpts);

    res.renderWithLocals('prices', data);
};

exports.list = function (req, res) {
    const root = helper.getBunkerNode(req.tld, req.bunker.sources);
    const pricesPage = _.get(root, ['prices-pages', 'list']);

    if (!pricesPage) {
        res.sendStatus(404);

        return;
    }

    const data = {
        page: 'prices',
        type: 'list',
        regions: pricesPage.regions
    };

    data.products = {
        title: pricesPage.title,
        text: pricesPage.text,
        data: pricesPage.data.map(block => ({
            footer: block.footer,
            list: getProductsPrices(block, 'list', root, req.pagesRelations, req.locals.url)
        }))
    };

    const seoOpts = {
        section: SECTION,
        titleData: _.get(data, 'products.title'),
        descriptionData: ''
    };

    data.seo = seo.getSeoData(req, seoOpts);

    res.renderWithLocals('prices', data);
};
