'use strict';

const _ = require('lodash');
const Filterable = require('./abstractions/filterable');
const helper = require('../lib/helper');
const Promise = require('bluebird');
const seo = require('../lib/seo');
const SolutionsModel = require('../model/solutions');

class Solutions extends Filterable {
    constructor(req, res, next) {
        super(req, res, next);

        this._filterParams = ['list', 'more', 'hasMore', 'permutation'];
    }

    index() {
        this._getSolutions({ section: '' });
    }

    stories() {
        this._getSolutions({ section: 'stories' });
    }

    cases() {
        this._getSolutions({ section: 'cases' });
    }

    practicums() {
        this._getSolutions({ section: 'practicums' });
    }

    analytics() {
        this._getSolutions({ section: 'analytics' });
    }

    story() {
        const { story } = this._req.params;

        this._getSolutions({
            section: 'stories',
            slug: story
        });
    }

    case() {
        const { caseArticle } = this._req.params;

        this._getSolutions({
            section: 'cases',
            slug: caseArticle
        });
    }

    practicum() {
        const { practicum } = this._req.params;

        this._getSolutions({
            section: 'practicums',
            slug: practicum
        });
    }

    analyticsArticle() {
        const { analyticsArticle } = this._req.params;

        this._getSolutions({
            section: 'analytics',
            slug: analyticsArticle
        });
    }

    others() {
        const { othersArticle } = this._req.params;

        this._getSolutions({
            section: 'others',
            slug: othersArticle
        });
    }

    filter() {
        const solutionsModel = new SolutionsModel(this._req, {
            ...this._req.body,
            type: 'solutions'
        });

        solutionsModel.fetch()
            .bind(this)
            .then(this._filterList)
            .catch(this._filterError);
    }

    analyticsNowIndex() {
        this._getSolutions({
            type: 'analytics-now',
            section: ''
        });
    }

    analyticsNowArticle() {
        const { selfArticle } = this._req.params;

        this._getSolutions({
            type: 'analytics-now',
            section: '_self',
            slug: selfArticle
        });
    }

    analyticsNowFilter() {
        const solutionsModel = new SolutionsModel(this._req, {
            ...this._req.body,
            type: 'analytics-now'
        });

        solutionsModel.fetch()
            .bind(this)
            .then(this._filterList)
            .catch(this._filterError);
    }

    _getSolutions(attributes) {
        const type = attributes.type || 'solutions';
        const page = attributes.slug ? 'solution' : 'solutions';

        SolutionsModel.CATEGORIES.forEach(category => {
            const value = this._req.query[category];

            if (value) {
                attributes.filter = (attributes.filter || []).concat({
                    by: category,
                    value: _.castArray(value)
                });
            }
        });

        const solutions = new SolutionsModel(this._req, attributes);

        return Promise
            .props({
                page,
                type,
                slug: attributes.slug,
                section: attributes.section,
                solutionsData: solutions.fetch(),
                filterUrl: this._buildAjaxPath(`${type}-filter`),
                solutionsUrl: this._buildPath(type),
                industry: this._req.query.industries,
                regionId: this._req.regionId
            })
            .then(this._addCta.bind(this))
            .then(this._addOgData.bind(this))
            .then(this._addSeoData.bind(this))
            .then(this._addLevels.bind(this))
            .then(this._addMetaData.bind(this))
            .then(this._renderSolutions.bind(this))
            .catch(this._renderError.bind(this));
    }

    _addCta(data) {
        const cta = _.get(data, 'solutionsData.solutionsPage.cta');

        if (cta) {
            return _.assign({ product: { cta } }, data);
        }

        return data;
    }

    _addOgData(data) {
        const solution = _.get(data, 'solutionsData.solution', {});
        const solutionOg = _.get(solution, 'og', {});

        const { block: blockOrder } = this._req.query;
        const foundBlockKey = _.findKey(solution, obj => {
            // Проверяем, что блок нужно показывать на странице, имеет шаблон (блок-этаж)
            // и совпадает порядок
            return _.get(obj, 'enabled', false)
                && _.has(obj, 'type')
                && _.get(obj, 'order') === Number(blockOrder);
        });
        const blockOg = _.get(solution, [foundBlockKey, 'og'], {});

        const page = _.get(data, 'solutionsData.solutionsPage');
        const foundIndustry = page && _(page.industries)
            .find(industry => industry.slug === data.industry);

        const industryOg = _.get(foundIndustry, 'og', {});
        const pageOg = _.assign(
            _.get(page, 'og', {}),
            solutionOg,
            blockOg
        );

        const defaultOg = helper.getBunkerNode(this._req.tld, this._req.bunker.settings, {
            path: 'og'
        });

        const resultOg = {
            title: industryOg.title || pageOg.title || defaultOg.title,
            description: industryOg.description || pageOg.description || defaultOg.description,
            image: industryOg.image || pageOg.image || defaultOg.image
        };

        return _.assign({ og: resultOg }, data);
    }

    _addSeoData(data) {
        const title = _.get(data, 'solutionsData.solutionsPage.title', '');
        const description = _.get(data, 'solutionsData.solutionsPage.text', '');
        const seoOpts = {
            section: data.type,
            titleData: _.get(data, 'solutionsData.solution.title', title),
            descriptionData: _.get(data, 'solutionsData.solution.description', description)
        };

        return _.assign({ seo: seo.getSeoData(this._req, seoOpts) }, data);
    }

    _addLevels(data) {
        const { solution } = data.solutionsData;

        return _.assign({
            levels: solution && helper.getPageLevels(solution),
            shareLevel: helper.getBunkerNode(this._req.tld,
                this._req.bunker.settings,
                {
                    path: 'share-level'
                })
        }, data);
    }

    _addMetaData(data) {
        const main = helper.getBunkerNode(
            this._req.tld,
            this._req.bunker.sources,
            { path: `${data.type}._main` }
        );

        return { ...data, regions: main && main.regions };
    }

    _renderSolutions(data) {
        const { type, page, section, solutionsData } = data;
        const redirectSection = section === 'others' ? '' : section;
        const redirectPath = this._buildPath(type, redirectSection);
        const dataStructure = _.get(solutionsData, data.slug ? 'solution' : 'solutionsPage');

        if (!_.isEmpty(dataStructure) && dataStructure.enabled) {
            if (dataStructure.react || !data.slug) {
                return this._res.reactRender({ data });
            }

            return this._res.renderWithLocals(page, data);
        }

        if (!_.isEmpty(dataStructure) && !dataStructure.enabled && dataStructure.redirect) {
            return this._res.redirect(301, dataStructure.redirect);
        }

        if (this._req.originalUrl === redirectPath) {
            if (redirectSection) {
                return this._res.redirect(301, this._buildPath(type));
            }

            return this._renderError();
        }

        this._res.redirect(301, redirectPath);
    }
}

module.exports = Solutions;
