'use strict';

const _ = require('lodash');
const url = require('url');
const config = require('yandex-cfg');
const Promise = require('bluebird');
const got = require('got');

const seo = require('../lib/seo');
const helper = require('../lib/helper');
const { buildPath } = require('../router/path');

const generateMessage = function (list) {
    return list
        .reduce((message, item) => {
            return item.value ? `${message}<br>${item.value}` : message;
        }, '');
};

const getSubscriptionPage = function (type, req, res) {
    const page = helper.getBunkerNode(req.tld, req.bunker.sources, {
        path: ['subscription-pages', type]
    });
    const pageName = 'subscription';

    if (!page) {
        res.sendStatus(404);

        return;
    }

    const seoOpts = {
        section: `${pageName}-${type}`,
        titleData: '',
        descriptionData: ''
    };

    let data = {
        page: 'subscription',
        subscriptionType: type,
        contactPageUrl: buildPath(req.locals.url, 'contact'),
        subscription: page,
        seo: seo.getSeoData(req, seoOpts),
        regions: page.regions
    };

    if (page.noindex) {
        data = _.set(data, 'meta.noindex', true);
    }

    res.renderWithLocals('subscription', data);
};

const getSubscriptionUrl = function (maillist, account) {
    return url.format({
        protocol: config.subscribe.protocol,
        hostname: config.subscribe.hostname,
        pathname: `${account}/maillist/${maillist}/subscription/request`
    });
};

const getTransactionUrl = maillist => url.format({
    protocol: config.subscribe.protocol,
    hostname: config.subscribe.hostname,
    pathname: `${config.subscribe.account}/transactional/${maillist}/send`
});

const requestToSender = function (data, mailLists, account) {
    /* eslint-disable camelcase */
    const options = {
        method: 'POST',
        auth: data.auth,
        json: true,
        body: {
            to_email: data.toEmail,
            params: JSON.stringify(data.params),
            key_subscription: data.key_subscription
        }
    };
    const successList = [];
    const requestsQueue = mailLists
        .map(mailList => got(getSubscriptionUrl(mailList.maillist, account), options)
            .then(() => {
                successList.push(mailList);
            })
            .catch(err => {
                err.failedMaillists = _.difference(mailLists, successList);

                return Promise.reject(err);
            }));

    return Promise.all(requestsQueue);
};

const captureSubscribeError = (req, res, error) => {
    if (error.internalCode) {
        req.logger.error(error, error.message);

        res
            .status(400)
            .send(error);
    } else {
        req.logger.error(error, 'Subscribe error');

        if (_.get(error, 'failedMaillists.0.value')) {
            res
                .status(400)
                .send({
                    internalCode: '400_SNA',
                    message: generateMessage(error.failedMaillists)
                });
        } else {
            res
                .status(400)
                .send({
                    internalCode: '400_SMF'
                });
        }
    }
};

/**
 * Подписка пользователя на рассылку в Рассыляторе
 * https://wiki.yandex-team.ru/users/selt/sender/double-opt-in/
 * @param {Object} req
 * @param {Object} res
 */
const sendSubscriptionRequest = function (req, res) {
    const fields = _.get(req, 'formData.fields.form', []);
    const subscriptionType = _.get(req, 'formData.fields.subscriptionType');
    const page = helper.getBunkerNode(req.tld, req.bunker.sources, {
        path: ['subscription-pages', subscriptionType]
    }) || {};
    const checkbox = _.find(page.fields, { type: 'checkbox' });
    const auth = config.subscribe.auth[subscriptionType] || config.subscribe.auth.default;
    const senderAccount = _.get(page, 'senderAccount', config.subscribe.account);

    const options = {
        toEmail: '',
        params: {},
        checkboxes: [],
        auth,
        key_subscription: 'adv'
    };

    /* eslint-disable complexity */
    // В пришедших данных находим E-mail, чекбоксы и логин
    fields.forEach(field => {
        if (field.isEmail) {
            options.toEmail = field.value;
        } else if (field.isCheckbox && field.checked) {
            options.checkboxes.push(field.value);
        } else if (field.value && field.name) {
            options.params[field.name] = field.value;
        }
    });
    /* eslint-enable complexity */

    //  Добавляем key_subscription если заполнено в бункере
    if (page.key_subscription) {
        options.params.key_subscription = page.key_subscription;
    }

    // Если не используем MailList по-умолчанию, то
    // находим у ноды чекбокса в Бункере значение MailList,
    // убираем дубликаты и строим список рассылок
    let mailLists = [];

    if (page.maillist) {
        mailLists = [{
            maillist: page.maillist,
            value: page.value
        }];
    } else {
        mailLists = _(options.checkboxes)
            .chain()
            .reduce((result, id) => {
                const checkboxData = _.find(checkbox.options, { id });

                result.push({
                    maillist: checkboxData.maillist,
                    value: checkboxData.value
                });

                return result;
            }, [])
            .uniq()
            .value();
    }

    if (!mailLists.length) {
        captureSubscribeError(req, res, {
            internalCode: '400_SNS',
            message: 'Subscriptions not selected'
        });

        return;
    }

    mailLists = helper.getSubscriptionMaillists(req, mailLists);

    requestToSender(options, mailLists, senderAccount)
        .then(() => res.sendStatus(204))
        .catch(error => captureSubscribeError(req, res, error));
};

const sendTransactionLetter = (req, res) => {
    const fields = _.get(req, 'formData.fields.form', []);
    const email = (_.find(fields, 'isEmail') || {}).value;

    if (!email) {
        captureSubscribeError(req, res, {
            internalCode: '400_ENS',
            message: 'Email is not specified'
        });

        return;
    }

    const options = {
        method: 'POST',
        auth: config.subscribe.auth.translation,
        query: {
            to_email: email
        }
    };

    got(getTransactionUrl(config.subscribe.transactionalMaillist), options)
        .then(() => res.sendStatus(204))
        .catch(error => captureSubscribeError(req, res, error));
};

exports.public = (req, res) => getSubscriptionPage('public', req, res);
exports.partners = (req, res) => getSubscriptionPage('partners', req, res);
exports.publishers = (req, res) => getSubscriptionPage('publishers', req, res);
exports.subs = (req, res) => getSubscriptionPage('subs', req, res);
exports.request = (req, res) => sendSubscriptionRequest(req, res);
exports.transaction = (req, res) => sendTransactionLetter(req, res);
