'use strict';

const _ = require('lodash');
const config = require('yandex-cfg');
const RSS = require('rss');

const helper = require('../lib/helper');
const seo = require('../lib/seo');

const PageModel = require('../model/page');
const Rss = require('./abstractions/rss');

process.env.NODE_TLS_REJECT_UNAUTHORIZED = '0';

class Turbo extends Rss {
    rss() {
        this._rssPath = 'turboRss';

        return this._getTurboPages()
            .then(this._sendXML.bind(this))
            .catch(this._renderError.bind(this));
    }

    _getTurboPages() {
        const feedData = this._getFeedData();
        const landingNode = helper.getBunkerNode(this._req.tld, this._req.bunker.sources, {
            path: 'landing'
        });

        _(landingNode)
            .keys()
            .each(pageName => {
                const page = new PageModel(this._req, {
                    section: pageName
                });

                page.getRss(feedData);
            });

        return this._getBlogsRss()
            .then(posts => {
                posts.forEach(post => {
                    this._getPostRss(post, feedData);
                });
            })
            .catch(error => {
                this._req.logger.error(error);
            })
            .then(() => feedData.xml({ indent: true }));
    }

    _getFeedData() {
        /* eslint-disable camelcase */
        const siteUrl = helper.getUrlByPathname(this._req, this._req.baseUrl);
        const seoData = seo.getSeoData(this._req, {});
        const customElements = [
            // TODO: унести хранение картинки в Бункер, удалить mordaLogo
            { 'turbo:logo': { _attr: { href: config.mordaLogo } } },
            { language: this._req.language },
            this._getMetrikaCounters()
        ];

        return new RSS({
            title: seoData.title,
            description: seoData.description,
            feed_url: `${siteUrl}/${this._rssPath}`,
            site_url: siteUrl,
            custom_namespaces: {
                turbo: 'http://turbo.yandex.ru',
                yandex: 'http://news.yandex.ru',
                media: 'http://search.yahoo.com/mrss/'
            },
            custom_elements: _(customElements)
                .flatten()
                .value()
        });
    }

    /**
     * Задаёт RSS для поста
     * @param {Object} post
     * @param {Object} feedData
     */
    _getPostRss(post, feedData) {
        const content = this._sanitizeTemplateStrings(post['turbo:content'][0]);

        feedData.item({
            _attr: { turbo: true },
            title: post.title[0],
            url: `${feedData.site_url}/news/${post.slug}`,
            date: post.pubDate[0],
            'custom_elements': [
                { _attr: { turbo: true } },
                { metrics: [{ yandex: { _attr: { schema_identifier: post.slug } } }] },
                { 'turbo:content': { _cdata: content } }
            ]
        });
    }

    _getMetrikaCounters() {
        return config.commonMetrikaCounters
            .map(counter => ({
                'yandex:analytics': {
                    _attr: {
                        id: counter.id,
                        type: 'Yandex'
                    }
                }
            }));
    }
}

module.exports = Turbo;
