const https = require('https');

const _ = require('lodash');
const config = require('yandex-cfg');
const got = require('got');

const defaultOptions = {
    agent: new https.Agent({ keepAlive: true }),
    json: true,
    headers: {
        'user-agent': `AdvertisingFrontend/${config.app.version}`
    }
};

class BackendError extends Error {
    constructor(originalError) {
        super(originalError.message);

        this.name = 'BackendError';
        this.originalError = originalError;

        this.code = originalError.code;
        this.statusCode = originalError.statusCode;
        this.responseHeaders = originalError.responseHeaders;
    }
}

class BackendClient {
    constructor(req) {
        this.tld = req.tld;
        this.tvmTicket = _.get(req, 'tvm.tickets.adv-backend.ticket');
    }

    companiesFilters(query) {
        return this.get('companies/filters', { query });
    }

    companiesSlugs(query) {
        return this.get('companies/slugs', { query });
    }

    companies(query) {
        return this.get('companies', { query });
    }

    editCompany(slug, data, login) {
        return this.patch(`companies/${slug}`, {
            body: {
                ...data,
                proposedBy: login
            },
            json: true,
            login
        });
    }

    get(path, options) {
        return this.request('get', path, options);
    }

    post(path, options) {
        return this.request('post', path, options);
    }

    patch(path, options) {
        return this.request('patch', path, options);
    }

    async request(method, pathname, options) {
        const { login, ...restOptions } = options;

        const url = `${config.backend.baseUrl}${pathname}/`;
        const opts = _.merge({
            headers: _.pickBy({
                'x-ya-service-ticket': this.tvmTicket,
                'x-ya-external-login': login
            }, Boolean),
            query: {
                tld: this.tld
            }
        }, defaultOptions, restOptions);

        try {
            const { body } = await got[method](url, opts);

            return body;
        } catch (error) {
            throw new BackendError(error);
        }
    }
}

module.exports = BackendClient;
