const _ = require('lodash');
const config = require('yandex-cfg');
const FormData = require('form-data');
const got = require('got');
const lru = require('lru-cache');
const Promise = require('bluebird');

const { project } = config.bunker;
const { baseUrl, paths, csrfTokenAge } = config.bunkerApi;

const cache = lru({ maxAge: csrfTokenAge });
const alias = 'csrf-token';

/**
 * Построение данных для POST-запроса
 * @param {Object} data
 * @returns {FormData}
 */
function buildFormData(data) {
    return _(data)
        .keys()
        .filter(key => Object.hasOwnProperty.call(data, key) && Boolean(data[key]))
        .transform((acc, key) => acc.append(key, data[key]), new FormData())
        .value();
}

/**
 * Получение CSRF-токена из ответа бункера
 * @param {String} response
 * @returns {String | undefined}
 */
function parseCSRFToken(response = '') {
    const match = response.match(/csrf-token&quot;:&quot;(.+?)&quot;/i);

    return _.get(match, 1);
}

/**
 * Хелперы для работы с API Бункера
 */
class BunkerApi {

    /**
     * @param {{ cookie: String?, csrfToken: String? }?} authData
     */
    constructor(authData = {}) {
        const { cookie, csrfToken } = authData;

        this._cookie = cookie;

        if (csrfToken) {
            cache.set(alias, csrfToken);
        }
    }

    /**
     * GET-запрос к API
     * @param {String} path
     * @param {String} node
     * @param {String?} version
     * @returns {Promise}
     * @private
     */
    _get(path, node, version = 'latest') {
        const options = { query: { node, version }, json: true };

        return got.get(`${config.bunker.api}${path}`, options)
            .then(data => ({
                body: _.get(data, 'body'),
                mime: data.headers['content-type']
            }))
            .catch(error => {
                switch (error.statusCode) {
                    case 404:
                        console.log('Нода не найдена', node);
                        break;
                    case 410:
                        console.log('Нода удалена', node);
                        break;
                    default:
                        throw error;
                }
            });
    }

    /**
     * POST-запрос к API
     * @param {String} path
     * @param {Object} options
     * @returns {Promise}
     * @private
     */
    _post(path, options = {}) {
        if (options.body) {
            options.body = buildFormData(options.body);
        }

        options.headers = _.assign({
            Cookie: this._cookie
        }, options.headers);

        return this._getCSRFToken()
            .then(token => {
                options.headers['X-CSRF-Token'] = token;

                return got.post(`${baseUrl}${path}`, options);
            })
            .then(response => JSON.parse(_.get(response, 'body')));
    }

    /**
     * Получает CSRF-токен, если он не был передан
     * @returns {Promise}
     * @private
     */
    _getCSRFToken() {
        if (cache.has(alias)) {
            return Promise.resolve(cache.get(alias));
        }

        const options = {
            headers: {
                Cookie: this._cookie
            }
        };

        return got.get(`${baseUrl}/${project}`, options)
            .then(response => parseCSRFToken(_.get(response, 'body')))
            .then(token => {
                cache.set(alias, token);

                return token;
            })
            .catch(error => {
                cache.del(alias);

                throw error;
            });
    }

    /**
     * Получение ноды из бункера
     * @param {String} node - путь до ноды в бункере
     * @param {{ version: String? }?} options - версия ноды. По умолчанию latest
     * @returns {Promise<Object?>}
     */
    get(node, options = {}) {
        return this._get(paths.cat, node, options.version);
    }

    /**
     * Получение всех поднод заданной ноды
     * @param {String} node - путь до ноды в бункере
     * @param {{ version: ('latest' | 'stable')?, recursive: Boolean? }?} options
     * @returns {Promise<Object?>}
     */
    list(node, options = {}) {
        const path = options.recursive ? paths.tree : paths.ls;

        return this._get(path, node, options.version)
            .then(data => data.body);
    }

    /**
     * Запись ноды в бункер
     * @param {String} node - путь до ноды в бункере
     * @param {JSON | Object} data - данные для записи
     * @param {String?} mime - схема ноды
     * @returns {Promise}
     */
    store(node, data, mime) {
        if (!this._cookie) {
            throw new Error('Для записи ноды требуются Cookie');
        }

        if (typeof data !== 'string') {
            data = JSON.stringify(data);
        }

        const options = {
            body: {
                'item[0]node': node,
                'item[0]mime': mime,
                'item[0]data': Buffer.from(data)
            }
        };

        return this._post(paths.store, options);
    }

    /**
     * Публикация ноды
     * @param {String} node - путь до ноды в бункере
     * @param {String} version - версия ноды. По умолчанию latest
     * @returns {Promise}
     */
    publish(node, version = 'latest') {
        if (!this._cookie) {
            throw new Error('Для публикации ноды требуются Cookie');
        }

        const options = {
            body: {
                'item[0]node': node,
                'item[0]version': version
            }
        };

        return this._post(paths.publish, options);
    }
}

module.exports = BunkerApi;
