'use strict';

const config = require('yandex-cfg');
const _ = require('lodash');
const { buildPath } = require('../router/path');
const env = process.env.NODE_ENV || require('yandex-environment');
const url = require('url');

const DATE_REGEXP = /(\d\d)\.(\d\d)\.(\d\d\d\d)/;

const HTML_MAPPINGS = {
    '&nbsp;': ' ',
    '&mdash;': '—',
    '&ndash;': '–',
    '&laquo;': '«',
    '&raquo;': '»'
};

const RE_HTML_ENTITIES = /&(?:nbsp|mdash|ndash|laquo|raquo);/g;
const RE_HAS_HTML_ENTITY = RegExp(RE_HTML_ENTITIES.source);


module.exports = {
    watchersByAgencyCountry(req, countryGeoId) {
        const watcherByCountries = _.get(req.bunker, ['watchers', 'agencies']);
        const countryData = _.find(watcherByCountries, data => data.geoId === countryGeoId);

        return countryData ? countryData.watchers : [];
    },

    /**
     * Определяет, поддерживается ли браузер
     * @param {Object} uatraits
     * @param {Object[]} supportedBrowsers
     * @returns {Boolean}
     */
    isBrowserValid: (uatraits, supportedBrowsers) => {
        if (!uatraits || !uatraits.BrowserVersion) {
            return false;
        }

        const [version] = uatraits.BrowserVersion.split('.');
        const supported = _.some(supportedBrowsers, browser => {
            const nameRegex = new RegExp(browser.name, 'i');

            return nameRegex.test(uatraits.BrowserName) && version >= browser.version;

        });

        return supported;
    },

    /**
     * Возвращает нужную ноду, основываясь на переданном tld.
     * Словарь соответствия tld и языковой папки лежит в config.bunkerLang.
     * @param {String} tld
     * @param {Object} root
     * @param {Object} [options]
     * @param {String|Array} [options.path]
     * @param {Boolean} [options.useRequestedTldOnly]
     * @returns {*}
     */
    /* eslint-disable complexity */
    getBunkerNode(tld, root, options) {
        let lang = config.bunkerLang[tld] || config.bunkerLang.default;
        const useRequestedTldOnly = _.get(options, 'useRequestedTldOnly');

        if (!Object.prototype.hasOwnProperty.call(root, lang) && !useRequestedTldOnly) {
            lang = config.bunkerLang.default;
        }

        if (!options || (_.isObject(options) && !options.path)) {
            return root[lang];
        }

        return _.get(root[lang], options.path || options);
    },
    /* eslint-enable complexity */

    /**
     * Функция, которая объединяет данные для переключателя локалей
     * Объединяет данные из поля 'regions' текущей модели и значения по умолчанию
     * data.regions заполняется в Бункере для каждой из страниц
     * @param {Object} sectionRegions - данные с редиректами из миддлвары section-region
     * @param {Object} pageRegions - данные с редиректами для конкретной страницы (из Бункера)
     * @param {Array<string>} regionsOrder - массив из tld, которые хотим показывать (порядок важен)
     * @returns {Array}
     */
    getRegions(sectionRegions, pageRegions, regionsOrder = config.regionsOrder) {
        return regionsOrder.map(regionId => {
            const sectionRegion = _.get(sectionRegions, regionId, {});
            const pageRegion = _.get(pageRegions, regionId, {});
            const regionData = config.regions[regionId];
            const path = pageRegion.path || sectionRegion.path || '/';

            return _.assign(regionData, {
                url: pageRegion.url || this.getTldBasePath(regionId, path),
                tld: regionId
            });
        });
    },

    getTldBasePath(tld, path) {
        const host = _.get(config.regionsHosts, tld, `${config.host}${tld}`);
        const pathname = buildPath(host, path);

        return `${host}${pathname}`;
    },

    /**
     * Возвращает язык для блога по tld
     * @param {string} tld
     * @returns {string}
     */
    getBlogLang(tld) {
        return config.tldToBlogLang[tld] || config.tldToBlogLang.default;
    },

    /**
     * Фильтруя поля переданного объекта, возвращает массив объектов, содержащих данные "этажей",
     * которые должны быть отображены на странице.
     * @param {Object} pageData Объект с данными страницы.
     * @param {Function | null} [additionalFilter] Функция-фильтр, получающая на вход объект
     * с данными одного "этажа" и возвращающая true, если данный "этаж" должен быть отображён на
     * странице, или false - если не должен.
     * Если данный параметр не задан, то в качестве фильтра используется функция,
     * всегда возвращающая true.
     * @returns {Object[]}
     */
    getPageLevels(pageData, additionalFilter) {
        return _(pageData)
            .values()
            .remove(level => {
                // Проверка, что уровень является объектом, имеет шаблон и флаг `enabled`
                const isEnabledObject = _.isObject(level) && _.has(level, 'type') && level.enabled;

                if (isEnabledObject) {
                    return additionalFilter instanceof Function ? additionalFilter(level) : true;
                }

                return false;
            })
            .sortBy('order')
            .value();
    },

    /**
     * Получает все слаги из поднод переданной ноды
     * @param {Object} page
     * @returns {String []}
     */
    getPageSlugs: page => {
        return _(page)
            .mapValues(node => (node.columns && _.flatten(node.columns)) || node.items || [])
            .values()
            .flatten()
            .map(item => item.link)
            .filter(item => item && !item.startsWith('http'))
            .value();
    },

    /**
     * Получает объект даты по переданной строке вида ДД.ММ.ГГГГ
     * @param {String} dateString
     * @returns {Date}
     */
    getDate(dateString) {
        const match = dateString.match(DATE_REGEXP);
        const month = match && Number(match[2]) - 1;

        return match && new Date(match[3], month, match[1]);
    },

    /**
     * Получает номер месяца п/п (1 - 12) по переданной строке в формате ДД.ММ.ГГГГ
     * @param {String} dateString
     * @returns {Number}
     */
    getMonth(dateString) {
        const match = dateString.match(DATE_REGEXP);

        return match && Number(match[2]);
    },

    /**
     * Возвращает заданный текст без мягких переносов &shy;
     * @param {String} text
     * @returns {String}
     */
    replaceShy(text) {
        return text.replace(/&shy;/gi, '');
    },

    /**
     * Удаляет слэш в конце урла
     * @param {String} path
     * @returns {String}
     */
    removeLastSlash(path) {
        return path.replace(/\/$/, '');
    },

    /**
     * Возвращает url по переданному пути
     * @param {Object} req
     * @param {String} pathname
     * @returns {String}
     */
    getUrlByPathname(req, pathname) {
        return url.format({
            protocol: 'https',
            host: req.headers.host,
            pathname
        });
    },

    /**
     * Функция сортировки дат в формате "Месяц ГГГГ"
     * @param {String} key
     * @param {String} order
     * @param {Object} entity
     * @returns {Number}
     */
    dateMonthYearSorter(key, order, entity = {}) {
        const orderSign = order === 'desc' ? -1 : 1;
        const value = entity[key];
        const [month, year] = value.split(' ');

        if (!month || !year) {
            return 0;
        }

        const monthNum = _.range(1, 13)
            .find(num => this.i18n('months', `n${num}`) === month) || 0;

        return orderSign * (new Date(parseInt(year, 10), monthNum - 1));
    },

    /**
     * Декодирует атрибуты, полученные из параметров запроса
     * @param {Object} attributes
     * @returns {Object}
     */
    decodeAttributes(attributes) {
        try {
            return _.transform(attributes, (acc, value, key) => {
                const rawValues = Array.isArray(value) ? value : [value];

                acc[key] = rawValues.map(decodeURIComponent);
            });
        } catch (e) {
            return {};
        }
    },

    /**
     * Возвращает этаж CTA по переданному тэгу статьи
     * @param {Object} req
     * @param {Object} tag
     * @returns {Object}
     * @private
     */
    getArticleCta(req, tag) {
        if (!tag) {
            return;
        }

        const articlesCta = module.exports
            .getBunkerNode(req.tld, req.bunker.sources, {
                path: 'articles-cta'
            });
        const levelCta = _(articlesCta)
            .values()
            .find(level => level.enabled && level.tag === tag.slug);

        if (!levelCta) {
            return;
        }

        levelCta.type = 'cta';

        return levelCta;
    },

    /**
     * Получает фамилию у сущности, имеющей поле name
     * @param {Object} named
     * @returns {String | undefined}
     */
    getSurname(named) {
        return named.name && _.get(named.name.split(' '), 1, named.name);
    },

    /**
     * Не в продакшне отправляем письма только на email'ы для тестирования
     * @param {Object} req,
     * @param {Object} recipient
     * @returns {Object}
     */
    getRecipient(req, recipient) {
        if (env !== 'production') {
            const settings = module.exports.getBunkerNode(req.tld, req.bunker.settings);

            return settings.testingEmails;
        }

        return recipient;
    },

    /**
     * Не в продакшне подписываемся на тестовую подписку
     * @param {Object} req,
     * @param {Object[]} maillists
     * @returns {Object}
     */
    getSubscriptionMaillists(req, maillists) {
        if (env !== 'production') {
            const settings = module.exports.getBunkerNode(req.tld, req.bunker.settings);

            return [settings.testingMailslist];
        }

        return maillists;
    },

    /**
     * Объединяет объекты в массивах по слагам, добавляет в элементы недостающие поля
     * @param {Object []} thisArray
     * @param {Object []} thatArray
     * @returns {Object []}
     * @private
     */
    joinArraysBySlug(thisArray, thatArray) {
        return thisArray.map(thisItem => {
            const sameItem = _(thatArray)
                .find(thatItem => thisItem.slug === thatItem.slug);

            return _.extend(thisItem, sameItem);
        });
    },

    /**
     * Убирает эмоджи основываясь на regex из https://github.com/mathiasbynens/emoji-regex
     * @param {String} str
     * @returns {String}
     */
    removeEmojis(str) {
        if (typeof str !== 'string') {
            return str;
        }

        /* eslint-disable max-len */
        const regex = /(?:[\u2700-\u27bf]|(?:\ud83c[\udde6-\uddff]){2}|[\ud800-\udbff][\udc00-\udfff]|[\u0023-\u0039]\ufe0f?\u20e3|\u3299|\u3297|\u303d|\u3030|\u24c2|\ud83c[\udd70-\udd71]|\ud83c[\udd7e-\udd7f]|\ud83c\udd8e|\ud83c[\udd91-\udd9a]|\ud83c[\udde6-\uddff]|[\ud83c[\ude01\uddff]|\ud83c[\ude01-\ude02]|\ud83c\ude1a|\ud83c\ude2f|[\ud83c[\ude32\ude02]|\ud83c\ude1a|\ud83c\ude2f|\ud83c[\ude32-\ude3a]|[\ud83c[\ude50\ude3a]|\ud83c[\ude50-\ude51]|\u203c|\u2049|[\u25aa-\u25ab]|\u25b6|\u25c0|[\u25fb-\u25fe]|\u00a9|\u00ae|\u2122|\u2139|\ud83c\udc04|[\u2600-\u26FF]|\u2b05|\u2b06|\u2b07|\u2b1b|\u2b1c|\u2b50|\u2b55|\u231a|\u231b|\u2328|\u23cf|[\u23e9-\u23f3]|[\u23f8-\u23fa]|\ud83c\udccf|\u2934|\u2935|[\u2190-\u21ff])/g;
        /* eslint-enable max-len */

        return str.replace(regex, '');
    },

    /**
     * Заменяет html мнемоники на символы юникода
     * @param {String} string
     * @returns {String}
     */
    replaceHtmlEntities(string) {
        return (string && RE_HAS_HTML_ENTITY.test(string))
            ? string.replace(RE_HTML_ENTITIES, entity => HTML_MAPPINGS[entity])
            : string;
    }
};
