/* eslint-disable new-cap */

const { erf, format } = require('mathjs');

function toRepresentation(x) {
    if (!Number.isFinite(x)) {
        return '';
    }

    return x < 1000
        ? format(x, {
            notation: 'auto',
            precision: 4,
            upperExp: 5,
            lowerExp: -5
        })
        : Math.round(x).toString();
}

class Segment {
    constructor({ size, cost, conversions, clicks, pValueTarget, k }) {
        // Размер сегмента
        this.size = size;

        // Расходы в сегменте
        this.cost = cost;

        // Конверсии
        this.conversions = conversions;

        // Клики
        this.clicks = clicks;

        // Целевое p-value
        this.pValueTarget = pValueTarget;

        // Максимально допустимый коэффициент жесткости дополнительных вложений
        this.k = k;
    }

    // CPA сегмента
    cpa() {
        return this.cost / this.conversions;
    }

    // Расходы относительно А
    costRelative(base) {
        return (this.cost * base.size) / (base.cost * this.size);
    }

    // Конверсии относительно А
    conversionsRelative(base) {
        return (this.conversions * base.size) / (base.conversions * this.size);
    }

    // Клики относительно А
    clicksRelative(base) {
        return (this.clicks * base.size) / (base.clicks * this.size);
    }

    // Относительный CPA
    cpaRelative(base) {
        return this.cpa() / base.cpa();
    }

    // Относительная погрешность расходов
    costErrorRelative() {
        return 1.5 * Math.sqrt(1 / this.clicks);
    }

    // Относительная погрешность конверсий
    conversionsErrorRelative() {
        return Math.sqrt(1 / this.conversions);
    }

    // Прибыль
    profit(base) {
        return (-1 * this.cost) + (this.k * base.cpa() * this.conversions);
    }

    // Относительная прибыль
    profitRelative(base) {
        return (this.profit(base) * base.size) / (base.profit(base) * this.size);
    }

    // Относительная погрешность прибыли
    profitErrorRelative(base) {
        return Math.sqrt(
            (Math.pow(this.cost * this.costErrorRelative(), 2)) +
            (
                Math.pow(this.k, 2) *
                Math.pow(base.cpa(), 2) *
                Math.pow(this.conversions * this.conversionsErrorRelative(), 2)
            )
        ) / this.profit(base);
    }

    // Статистическая значимость
    statValue(base) {
        return (
            (this.profitRelative(base) - base.profitRelative(base)) /
            Math.sqrt(
                (
                    Math.pow(this.profitErrorRelative(base), 2) *
                    Math.pow(this.profitRelative(base), 2) /
                    Math.pow(base.profitRelative(base), 2)
                ) + (Math.pow(base.profitErrorRelative(base), 2)))
        );
    }

    // Вероятность того, что B лучше A
    pValue(base) {
        return 0.5 * (1 + erf(this.statValue(base) / Math.sqrt(2)));
    }

    // Используем?
    shipIt(base) {
        const pValue = this.pValue(base) * 100;

        if (pValue > this.pValueTarget) {
            return 'YES';
        }

        if (100 - pValue < this.pValueTarget) {
            return 'UNDEF';
        }

        return 'NO';
    }
}

class SegmentsComparison {
    constructor({ pValueTarget, k }) {
        // Целевое p-value
        this.pValueTarget = pValueTarget;

        // Максимально допустимый коэффициент жесткости дополнительных вложений
        this.k = k;

        // Сегменты для сравнения
        this.segments = [];
    }

    addSegment(segment) {
        const { pValueTarget, k } = this;

        this.segments.push(
            new Segment({ ...segment, pValueTarget, k })
        );

        return this;
    }

    compare() {
        const [baseSegment] = this.segments;

        return this.segments.map(segment => ({
            cpa: toRepresentation(segment.cpa()),
            costRelative: toRepresentation(segment.costRelative(baseSegment) * 100),
            conversionsRelative: toRepresentation(segment.conversionsRelative(baseSegment) * 100),
            clicksRelative: toRepresentation(segment.clicksRelative(baseSegment) * 100),
            cpaRelative: toRepresentation(segment.cpaRelative(baseSegment) * 100),
            costErrorRelative: toRepresentation(segment.costErrorRelative() * 100),
            conversionsErrorRelative: toRepresentation(segment.conversionsErrorRelative() * 100),
            profit: toRepresentation(segment.profit(baseSegment)),
            profitRelative: toRepresentation(segment.profitRelative(baseSegment) * 100),
            profitErrorRelative: toRepresentation(segment.profitErrorRelative(baseSegment) * 100),
            statValue: toRepresentation(segment.statValue(baseSegment)),
            pValue: toRepresentation(segment.pValue(baseSegment) * 100),
            shipIt: segment.shipIt(baseSegment)
        }));
    }
}

module.exports = { SegmentsComparison };
