const _ = require('lodash');
const config = require('yandex-cfg');
const helper = require('../lib/helper');
const { buildPath, buildAjaxPath } = require('../router/path');
const { isBrowserValid } = require('../lib/helper');
const url = require('url');

/**
 * Возвращает префутер
 * @param {Object} req
 * @returns {Object}
 */
function getPrefooter(req) {
    return helper.getBunkerNode(req.tld, req.bunker.menu.prefooter);
}

/**
 * Возвращает объект blackbox, удаляя чувствительные данные
 * @param {Object} req
 * @returns {Object}
 */
function getBlackbox(req) {
    const blackbox = _.omit(req.blackbox, ['userTicket']);

    blackbox.raw = _.omit(blackbox.raw, ['user_ticket']);

    return blackbox;
}

/**
 * Возвращает объект userInfo
 * @param {Object} blackbox
 * @param {String} domain
 * @returns {Object}
 */
function getUserInfo(blackbox, domain) {
    const userInfo = { isAuthorized: false };

    if (blackbox.status === 'VALID') {
        const userPhones = _.get(blackbox, 'raw.phones');

        userInfo.uid = _.get(blackbox, 'raw.uid.value');
        userInfo.displayName = _.get(blackbox, 'displayName');
        userInfo.login = blackbox.login;
        userInfo.isAuthorized = true;
        userInfo.email = _.get(blackbox, 'raw.address-list.0.address');
        userInfo.shouldBoundPhone = domain === 'by' && _.isEmpty(userPhones);
    }

    return userInfo;
}

module.exports = (req, res, next) => {
    const domain = req.tld;
    const blackbox = getBlackbox(req);
    const userInfo = getUserInfo(blackbox, domain);
    const pathAdroll = config.sources.url + config.sources.adroll.path;
    const searchid = config.searchIds[req.tld];

    const urlPrefix = buildPath(req.currentUrl);
    // eslint-disable-next-line no-undefined
    const basename = urlPrefix === '/' ? undefined : urlPrefix;

    req.locals = {
        version: config.app.version,
        stHost: config.static.host,
        stNoCache: config.static.noCache,
        debug: config.app.debug,
        host: req.headers.host,
        language: req.language,
        contentRegion: domain.split('.').pop(),
        tld: domain,
        basename,
        uatraits: req.uatraits,
        isTouch: req.uatraits.isTouch,
        nonce: req.nonce,
        blackbox,
        userInfo,
        secretKey: req.secretkey,
        yandexuid: _.escape(req.cookies.yandexuid),
        prefooter: getPrefooter(req),
        settings: _.pick(helper.getBunkerNode(domain, req.bunker.settings),
            ['footer', 'dashboardHelp']),
        meta: helper.getBunkerNode(domain, req.bunker.settings, {
            path: 'meta'
        }),
        isDebug: ['local', 'testing']
            .indexOf(process.env.NODE_ENV) > -1 && req.query.debug,
        formsUrl: config.forms.url,
        query: req.query,
        contactFormPath: buildAjaxPath(req.currentUrl, config.forms.contactUrl),
        subscribeFormPath: buildAjaxPath(req.currentUrl, config.forms.subscribeUrl),
        homeUrl: buildPath(req.currentUrl, '/'),
        transactionPath: buildAjaxPath(req.currentUrl, config.forms.transactionPath),
        faviconUrl: config.favicon[domain] || config.favicon.default,
        currentPath: buildPath(req.currentUrl, req.path.replace(/^\/adv\//, '/')),
        countryId: config.geoIdByTld[domain],
        url: req.currentUrl,
        mordaHost: url.format({
            protocol: config.morda.protocol,
            host: config.morda[domain].host
        }),
        urlBrowserUpdater: config.browserUpdater,
        adroll: config.sources.adroll.languages.indexOf(req.tld) > -1 && pathAdroll,
        metrikaCounters: config.commonMetrikaCounters
            .filter(counter => !counter.tld || _.includes(counter.tld, domain)),
        metrikaCountersParams: config.metrikaCountersParams,
        passportHost: url.format({
            protocol: config.passport.protocol,
            host: config.passport.host[domain]
        }),
        addUserUrl: url.format({
            protocol: config.passport.protocol,
            host: config.passport.host[domain],
            pathname: config.passport.paths.auth,
            query: {
                mode: 'add-user',
                retpath: req.currentUrl
            }
        }),
        shareScript: config.share,
        siteSearchUrl: searchid && url.format({
            protocol: 'https',
            host: req.headers.host,
            pathname: buildPath(req.currentUrl, 'search'),
            query: { searchid }
        }),
        shareServices: _.get(config.shareServices, req.tld, 'default'),
        newsPage: buildPath(req.currentUrl, 'news'),
        journalsPage: url.format({
            protocol: 'https',
            host: req.headers.host,
            pathname: buildPath(req.currentUrl, 'journals')
        }),
        refreshCaptchaUrl: buildAjaxPath(req.currentUrl, 'captcha/new'),
        comdiUrl: `${config.sources.url}${config.sources.comdi.path}`,
        isBrowserValid: isBrowserValid(req.uatraits, config.comdi.supportedBrowsers),
        reactScript: config.reactScript,
        htmlMappings: config.htmlMappings,
        agenciesConfig: _.pick(config.agencies, ['groupsByCountryGeoId', 'agency'])
    };

    res.renderWithLocals = function (bundleName, data, block) {
        data.meta = _.merge({}, _.get(req, 'locals.meta'), data.meta);
        data.regions = helper.getRegions(req.regions, data.regions);

        const platformBundle = req.uatraits.isMobile
            ? 'touch-phone'
            : 'desktop';

        res.bundleRender(bundleName, _.assign({}, req.locals, data), platformBundle, block);
    };

    res.reactRender = function ({ bundleName, block, data = {}, state = {} }) {
        const meta = _.merge({}, _.get(req, 'locals.meta'), data.meta);
        const regions = helper.getRegions(req.regions, data.regions, req.regionsOrder);

        const platformBundle = req.uatraits.isMobile
            ? 'touch'
            : 'desktop';

        const stateData = {
            ...state,
            sidebar: {
                menuIndexChain: (req.locals.breadcrumbsChain || []).map(Number),
                isOpen: _.get(req.locals, 'mainMenu.openMenu', false)
            }
        };

        const appData = { block, ...req.locals, ...data, meta, regions };

        res.renderBundle(bundleName || platformBundle, appData, stateData);
    };

    next();
};
